import React, { useMemo, useState } from "react";
import { useAuth } from "@/contexts/AuthContext";
import { useLocation } from "wouter";
import { Button } from "@/components/ui/button";

const Check = (props: React.SVGProps<SVGSVGElement>) => (
  <svg
    viewBox="0 0 24 24"
    fill="none"
    aria-hidden="true"
    {...props}
    className={"w-5 h-5 " + (props.className ?? "")}
  >
    <path d="M20 6L9 17l-5-5" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" />
  </svg>
);

export default function Pricing() {
  const { currentUser } = useAuth();
  const [, setLocation] = useLocation();
  const [billing, setBilling] = useState<"monthly" | "yearly">("monthly");

  const prices = useMemo(
    () => ({
      free: { amount: "$0", unit: "/forever" },
      pro:
        billing === "monthly"
          ? { amount: "$19", unit: "/month" }
          : { amount: "$190", unit: "/year" },
      enterprise: { amount: "Custom", unit: "/contact us" },
    }),
    [billing]
  );

  const handleProUpgrade = () => {
    if (!currentUser) {
      setLocation(`/signup?intent=pro&billing=${billing}`);
      return;
    }
    // Handle payment modal for logged-in users
  };

  return (
    <section id="pricing" className="py-16 bg-white dark:bg-gray-800">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center mb-12">
          <h2 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-4">
            Simple, Transparent Pricing
          </h2>
          <p className="text-lg text-muted-foreground max-w-3xl mx-auto">
            Choose the perfect plan for your business needs
          </p>
        </div>

        {/* Billing Toggle */}
        <div className="flex justify-center mb-8" role="group" aria-label="Billing cycle">
          <div className="bg-gray-100 dark:bg-gray-800 p-1 rounded-lg">
            <button
              onClick={() => setBilling("monthly")}
              className={`px-4 py-2 text-sm font-medium rounded transition-colors ${
                billing === "monthly"
                  ? "bg-white dark:bg-gray-700 text-gray-900 dark:text-white shadow-sm"
                  : "text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white"
              }`}
              aria-pressed={billing === "monthly"}
            >
              Monthly
            </button>
            <button
              onClick={() => setBilling("yearly")}
              className={`px-4 py-2 text-sm font-medium rounded transition-colors ${
                billing === "yearly"
                  ? "bg-white dark:bg-gray-700 text-gray-900 dark:text-white shadow-sm"
                  : "text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white"
              }`}
              aria-pressed={billing === "yearly"}
            >
              Yearly
            </button>
          </div>
        </div>

        {/* Pricing Cards */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-8 max-w-5xl mx-auto">
          {/* Free Plan */}
          <div className="border rounded-xl p-6 bg-white dark:bg-gray-900">
            <h3 className="text-xl font-semibold mb-2">Free</h3>
            <div className="mb-4">
              <span className="text-3xl font-bold">{prices.free.amount}</span>
              <span className="text-gray-600 dark:text-gray-400">{prices.free.unit}</span>
            </div>
            <ul className="space-y-3 mb-6">
              <li className="flex items-center gap-3">
                <Check className="text-green-500" />
                <span className="text-sm">Basic name generation</span>
              </li>
              <li className="flex items-center gap-3">
                <Check className="text-green-500" />
                <span className="text-sm">5 brand kit downloads</span>
              </li>
              <li className="flex items-center gap-3">
                <Check className="text-green-500" />
                <span className="text-sm">Standard templates</span>
              </li>
            </ul>
            <Button variant="outline" className="w-full">
              Get Started Free
            </Button>
          </div>

          {/* Pro Plan */}
          <div className="border-2 border-primary rounded-xl p-6 bg-white dark:bg-gray-900 relative">
            <div className="absolute -top-3 left-1/2 -translate-x-1/2 bg-primary text-white px-3 py-1 rounded-full text-sm font-medium">
              Most Popular
            </div>
            <h3 className="text-xl font-semibold mb-2">Pro</h3>
            <div className="mb-4">
              <span className="text-3xl font-bold">{prices.pro.amount}</span>
              <span className="text-gray-600 dark:text-gray-400">{prices.pro.unit}</span>
            </div>
            <ul className="space-y-3 mb-6">
              <li className="flex items-center gap-3">
                <Check className="text-green-500" />
                <span className="text-sm">Unlimited name generation</span>
              </li>
              <li className="flex items-center gap-3">
                <Check className="text-green-500" />
                <span className="text-sm">Unlimited brand kit downloads</span>
              </li>
              <li className="flex items-center gap-3">
                <Check className="text-green-500" />
                <span className="text-sm">Premium templates access</span>
              </li>
              <li className="flex items-center gap-3">
                <Check className="text-green-500" />
                <span className="text-sm">1 free domain per year</span>
              </li>
              <li className="flex items-center gap-3">
                <Check className="text-green-500" />
                <span className="text-sm">Priority support</span>
              </li>
            </ul>
            <Button onClick={handleProUpgrade} className="w-full bg-primary hover:bg-primary/90">
              Upgrade to Pro
            </Button>
          </div>

          {/* Enterprise Plan */}
          <div className="border rounded-xl p-6 bg-white dark:bg-gray-900">
            <h3 className="text-xl font-semibold mb-2">Enterprise</h3>
            <div className="mb-4">
              <span className="text-3xl font-bold">{prices.enterprise.amount}</span>
              <span className="text-gray-600 dark:text-gray-400">{prices.enterprise.unit}</span>
            </div>
            <ul className="space-y-3 mb-6">
              <li className="flex items-center gap-3">
                <Check className="text-green-500" />
                <span className="text-sm">Everything in Pro</span>
              </li>
              <li className="flex items-center gap-3">
                <Check className="text-green-500" />
                <span className="text-sm">White-label solutions</span>
              </li>
              <li className="flex items-center gap-3">
                <Check className="text-green-500" />
                <span className="text-sm">API access</span>
              </li>
              <li className="flex items-center gap-3">
                <Check className="text-green-500" />
                <span className="text-sm">Dedicated account manager</span>
              </li>
            </ul>
            <Button variant="outline" className="w-full">
              Contact Sales
            </Button>
          </div>
        </div>
      </div>
    </section>
  );
}