import React from 'react';
import { useAuth } from '@/contexts/AuthContext';
import { PaywallModal } from '@/components/PaywallModal';
import { AuthModal } from '@/components/AuthModal';
import { useLocation } from 'wouter';

/**
 * Higher-Order Component that wraps components requiring Pro subscription
 * Shows paywall modal for non-paid users, returns wrapped component for paid users
 */
export function withPro<P extends object>(
  Component: React.ComponentType<P>,
  options?: {
    feature?: string;
    description?: string;
  }
) {
  return function WrappedWithPro(props: P) {
    const { currentUser, loading, isPaidUser, isOwnerUser } = useAuth();
    const [, navigate] = useLocation();

    // Show loading state while auth is loading
    if (loading) {
      return (
        <div className="flex items-center justify-center min-h-[200px]">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
        </div>
      );
    }

    // FIXED: Gate unauthenticated users first
    if (!currentUser) {
      return (
        <AuthModal
          isOpen={true}
          onClose={() => navigate('/')}
        />
      );
    }

    // FIXED: Gate authenticated but non-paid users (owners bypass paywall)
    if (!isPaidUser && !isOwnerUser) {
      return (
        <PaywallModal
          isOpen={true}
          onClose={() => navigate('/pricing')}
          feature={options?.feature || "premium features"}
          description={options?.description || "Access advanced tools and capabilities"}
        />
      );
    }

    // Only paid authenticated users get the component
    return <Component {...props} />;
  };
}

// Convenience HOC for common use cases
export function withBusinessPlanPro<P extends object>(Component: React.ComponentType<P>) {
  return withPro(Component, {
    feature: "business plan exports",
    description: "Export your business plans as PDF, DOCX, or save to Google Docs"
  });
}

export function withBrandKitPro<P extends object>(Component: React.ComponentType<P>) {
  return withPro(Component, {
    feature: "brand kit generation",
    description: "Generate complete brand kits with logos, colors, and assets"
  });
}

export function withExportPro<P extends object>(Component: React.ComponentType<P>) {
  return withPro(Component, {
    feature: "export and download",
    description: "Export and download your creations in multiple formats"
  });
}