// Visual Template Renderers - Convert visual data to markdown for business plan sections

import type { SwotData, PersonaData, TamSamSomData, PricingTierData } from "@shared/visuals";

/**
 * Render SWOT Analysis data as markdown
 */
export function renderSwotMarkdown(data: SwotData): string {
  const { strengths, weaknesses, opportunities, threats } = data;
  
  return `## SWOT Analysis

### Strengths
${strengths.length > 0 ? strengths.map(item => `- ${item}`).join('\n') : '- *No strengths defined yet*'}

### Weaknesses  
${weaknesses.length > 0 ? weaknesses.map(item => `- ${item}`).join('\n') : '- *No weaknesses defined yet*'}

### Opportunities
${opportunities.length > 0 ? opportunities.map(item => `- ${item}`).join('\n') : '- *No opportunities defined yet*'}

### Threats
${threats.length > 0 ? threats.map(item => `- ${item}`).join('\n') : '- *No threats defined yet*'}

---
*SWOT Analysis generated on ${new Date().toLocaleDateString()}*`;
}

/**
 * Render Customer Persona data as markdown
 */
export function renderPersonaMarkdown(data: PersonaData): string {
  const { name, demographics, goals, frustrations, buyingTriggers } = data;
  
  return `## Customer Persona${name ? `: ${name}` : ''}

### Demographics
${demographics || '*No demographics defined yet*'}

### Goals & Objectives
${goals || '*No goals defined yet*'}

### Frustrations & Pain Points
${frustrations || '*No frustrations defined yet*'}

### Buying Triggers & Motivators
${buyingTriggers || '*No buying triggers defined yet*'}

---
*Customer Persona created on ${new Date().toLocaleDateString()}*`;
}

/**
 * Render TAM/SAM/SOM market sizing data as markdown
 */
export function renderTamSamSomMarkdown(data: TamSamSomData): string {
  const { tam, sam, som, sources } = data;
  
  return `## Market Sizing Analysis (TAM/SAM/SOM)

### TAM (Total Addressable Market)
${tam || '*Total addressable market size not defined yet*'}

### SAM (Serviceable Available Market)  
${sam || '*Serviceable available market size not defined yet*'}

### SOM (Serviceable Obtainable Market)
${som || '*Serviceable obtainable market size not defined yet*'}

### Sources & Assumptions
${sources || '*No sources or assumptions documented yet*'}

---
*Market sizing analysis generated on ${new Date().toLocaleDateString()}*`;
}

/**
 * Render Pricing Table data as markdown
 */
export function renderPricingMarkdown(tiers: PricingTierData[]): string {
  if (!tiers.length) {
    return `## Pricing Strategy

*No pricing tiers defined yet*

---
*Pricing table generated on ${new Date().toLocaleDateString()}*`;
  }
  
  const tableHeader = '| Tier | Price | Features |';
  const tableSeparator = '|------|-------|----------|';
  const tableRows = tiers.map(tier => {
    const featuresText = tier.features.length > 0 
      ? tier.features.join(', ')
      : 'No features defined';
    return `| ${tier.name} | ${tier.price} | ${featuresText} |`;
  });
  
  const featuresList = tiers.map(tier => {
    if (tier.features.length === 0) return '';
    return `\n### ${tier.name} Features\n${tier.features.map(f => `- ${f}`).join('\n')}`;
  }).filter(Boolean).join('\n');
  
  return `## Pricing Strategy

${tableHeader}
${tableSeparator}
${tableRows.join('\n')}
${featuresList}

---
*Pricing table generated on ${new Date().toLocaleDateString()}*`;
}

/**
 * Utility function to clean and format markdown before insertion
 */
export function cleanMarkdownForInsertion(markdown: string): string {
  return markdown
    .trim()
    .replace(/^\n+|\n+$/g, '') // Remove leading/trailing newlines
    .replace(/\n{3,}/g, '\n\n'); // Normalize multiple newlines to double
}

/**
 * Get a preview of rendered markdown (first 150 characters)
 */
export function getMarkdownPreview(markdown: string): string {
  const cleaned = cleanMarkdownForInsertion(markdown);
  if (cleaned.length <= 150) return cleaned;
  return cleaned.substring(0, 147) + '...';
}

/**
 * Validate visual data before rendering
 */
export function validateSwotData(data: SwotData): string[] {
  const errors: string[] = [];
  if (!data.strengths || !Array.isArray(data.strengths)) {
    errors.push('Strengths must be an array');
  }
  if (!data.weaknesses || !Array.isArray(data.weaknesses)) {
    errors.push('Weaknesses must be an array');
  }
  if (!data.opportunities || !Array.isArray(data.opportunities)) {
    errors.push('Opportunities must be an array');
  }
  if (!data.threats || !Array.isArray(data.threats)) {
    errors.push('Threats must be an array');
  }
  return errors;
}

export function validatePersonaData(data: PersonaData): string[] {
  const errors: string[] = [];
  if (typeof data.name !== 'string') errors.push('Name must be a string');
  if (typeof data.demographics !== 'string') errors.push('Demographics must be a string');
  if (typeof data.goals !== 'string') errors.push('Goals must be a string');
  if (typeof data.frustrations !== 'string') errors.push('Frustrations must be a string');
  if (typeof data.buyingTriggers !== 'string') errors.push('Buying triggers must be a string');
  return errors;
}

export function validateTamSamSomData(data: TamSamSomData): string[] {
  const errors: string[] = [];
  if (typeof data.tam !== 'string') errors.push('TAM must be a string');
  if (typeof data.sam !== 'string') errors.push('SAM must be a string');
  if (typeof data.som !== 'string') errors.push('SOM must be a string');
  if (typeof data.sources !== 'string') errors.push('Sources must be a string');
  return errors;
}

export function validatePricingData(tiers: PricingTierData[]): string[] {
  const errors: string[] = [];
  if (!Array.isArray(tiers)) {
    errors.push('Pricing tiers must be an array');
    return errors;
  }
  
  tiers.forEach((tier, index) => {
    if (typeof tier.name !== 'string') {
      errors.push(`Tier ${index + 1}: Name must be a string`);
    }
    if (typeof tier.price !== 'string') {
      errors.push(`Tier ${index + 1}: Price must be a string`);
    }
    if (!Array.isArray(tier.features)) {
      errors.push(`Tier ${index + 1}: Features must be an array`);
    }
  });
  
  return errors;
}