// SVG sanitizer and normalizer for AI-generated logos
// Strips dangerous content and ensures proper scaling

export function sanitizeAndNormalizeSvg(svgString: string): { svg: string; warnings: string[] } {
  const warnings: string[] = [];
  
  try {
    // Parse SVG
    const parser = new DOMParser();
    const doc = parser.parseFromString(svgString, 'image/svg+xml');
    const svg = doc.querySelector('svg');
    
    if (!svg) {
      throw new Error('Invalid SVG: no <svg> element found');
    }
    
    // Remove dangerous elements
    const dangerousSelectors = [
      'script',
      'foreignObject',
      'iframe',
      'object',
      'embed'
    ];
    
    dangerousSelectors.forEach(selector => {
      const elements = svg.querySelectorAll(selector);
      elements.forEach(el => {
        el.remove();
        warnings.push(`Removed ${selector} element for security`);
      });
    });
    
    // Remove event handlers (on* attributes)
    const allElements = svg.querySelectorAll('*');
    allElements.forEach(el => {
      Array.from(el.attributes).forEach(attr => {
        if (attr.name.startsWith('on')) {
          el.removeAttribute(attr.name);
          warnings.push(`Removed event handler: ${attr.name}`);
        }
      });
    });
    
    // Remove external references (xlink:href, href pointing to http/https)
    allElements.forEach(el => {
      const href = el.getAttribute('href') || el.getAttribute('xlink:href');
      if (href && (href.startsWith('http://') || href.startsWith('https://') || href.startsWith('//'))) {
        el.removeAttribute('href');
        el.removeAttribute('xlink:href');
        warnings.push(`Removed external reference: ${href.substring(0, 50)}...`);
      }
    });
    
    // Remove text elements (we want icon-only)
    const textElements = svg.querySelectorAll('text, tspan');
    if (textElements.length > 0) {
      textElements.forEach(el => el.remove());
      warnings.push('Removed text elements - this generator produces icon-only SVGs');
    }
    
    // Helper to check if color is white or near-white
    const isWhiteColor = (color: string | null): boolean => {
      if (!color) return false;
      const c = color.toLowerCase().trim();
      
      // Pure white
      if (c === '#fff' || c === '#ffffff' || c === 'white') return true;
      
      // RGB/RGBA white or near-white (>240 for all channels)
      const rgbMatch = c.match(/rgba?\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)/i);
      if (rgbMatch) {
        const [, r, g, b] = rgbMatch.map(Number);
        return r >= 240 && g >= 240 && b >= 240;
      }
      
      return false;
    };
    
    // Remove white/near-white fill from SVG root element
    const svgFill = svg.getAttribute('fill');
    if (isWhiteColor(svgFill)) {
      svg.removeAttribute('fill');
      warnings.push('Removed white fill from SVG root');
    }
    
    // Remove background styling from SVG root
    const svgStyle = svg.getAttribute('style');
    if (svgStyle && /background\s*:/i.test(svgStyle)) {
      const cleanedStyle = svgStyle.replace(/background[^;]*;?/gi, '').trim();
      if (cleanedStyle) {
        svg.setAttribute('style', cleanedStyle);
      } else {
        svg.removeAttribute('style');
      }
      warnings.push('Removed background style from SVG root');
    }
    
    // Remove <style> blocks with background or white fill rules
    const styleElements = svg.querySelectorAll('style');
    styleElements.forEach(style => {
      const content = style.textContent || '';
      if (/background\s*:/i.test(content) || /fill\s*:\s*(#fff|#ffffff|white)/i.test(content)) {
        style.remove();
        warnings.push('Removed <style> block with background/white fill rules');
      }
    });
    
    // Remove embedded raster images
    const imageElements = svg.querySelectorAll('image');
    if (imageElements.length > 0) {
      imageElements.forEach(img => img.remove());
      warnings.push(`Removed ${imageElements.length} embedded raster image(s)`);
    }
    
    // Remove white/near-white background shapes (paths, polygons, rects)
    const backgroundShapes = svg.querySelectorAll('rect, path, polygon, circle, ellipse');
    backgroundShapes.forEach((el, index) => {
      const fill = el.getAttribute('fill');
      const tagName = el.tagName.toLowerCase();
      
      // Check if it's white or near-white
      if (isWhiteColor(fill)) {
        // First element is almost always background - remove it
        if (index === 0) {
          el.remove();
          warnings.push(`Removed white/near-white background ${tagName} (fill: ${fill})`);
        }
        // Also check if it's in a background group
        else if (el.parentElement?.id?.toLowerCase().includes('background')) {
          el.remove();
          warnings.push(`Removed white/near-white ${tagName} from background group`);
        }
      }
    });
    
    // Ensure viewBox exists
    if (!svg.hasAttribute('viewBox')) {
      const width = parseFloat(svg.getAttribute('width') || '100');
      const height = parseFloat(svg.getAttribute('height') || '100');
      svg.setAttribute('viewBox', `0 0 ${width} ${height}`);
      warnings.push('Added missing viewBox attribute');
    }
    
    // Remove fixed width/height so it scales properly
    if (svg.hasAttribute('width') || svg.hasAttribute('height')) {
      svg.removeAttribute('width');
      svg.removeAttribute('height');
    }
    
    // Ensure xmlns is present
    if (!svg.hasAttribute('xmlns')) {
      svg.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
    }
    
    // Serialize back to string
    const serializer = new XMLSerializer();
    const cleanSvg = serializer.serializeToString(svg);
    
    return { svg: cleanSvg, warnings };
    
  } catch (error) {
    throw new Error(`Failed to sanitize SVG: ${error instanceof Error ? error.message : 'Unknown error'}`);
  }
}

// Helper to check if SVG is valid before sanitizing
export function isValidSvg(svgString: string): boolean {
  try {
    const parser = new DOMParser();
    const doc = parser.parseFromString(svgString, 'image/svg+xml');
    const parserError = doc.querySelector('parsererror');
    return !parserError && !!doc.querySelector('svg');
  } catch {
    return false;
  }
}
