import type { StructuredTemplate } from "@shared/plan-types";

export const SWOT_TEMPLATE: StructuredTemplate = {
  key: "swot",
  name: "SWOT Analysis",
  description: "Strengths, Weaknesses, Opportunities, Threats.",
  fields: [
    { id: "strengths", label: "Strengths", type: "textarea", placeholder: "List your company's internal strengths..." },
    { id: "weaknesses", label: "Weaknesses", type: "textarea", placeholder: "List your company's internal weaknesses..." },
    { id: "opportunities", label: "Opportunities", type: "textarea", placeholder: "List external opportunities..." },
    { id: "threats", label: "Threats", type: "textarea", placeholder: "List external threats..." },
  ],
};

export const PERSONA_TEMPLATE: StructuredTemplate = {
  key: "persona",
  name: "Customer Persona",
  description: "Capture ICP/persona details.",
  fields: [
    { id: "name", label: "Persona Name", type: "text", placeholder: "e.g., Side-Hustle Sarah" },
    { id: "demographics", label: "Demographics", type: "textarea", placeholder: "Age, location, income, education..." },
    { id: "jobs", label: "Jobs-To-Be-Done / Goals", type: "textarea", placeholder: "What are they trying to accomplish?" },
    { id: "pain", label: "Pain Points", type: "textarea", placeholder: "What frustrates them or blocks their goals?" },
    { id: "channels", label: "Acquisition Channels", type: "textarea", placeholder: "Where do they discover new solutions?" },
  ],
};

export const V2_TEMPLATES: StructuredTemplate[] = [
  SWOT_TEMPLATE,
  PERSONA_TEMPLATE,
  {
    key: "porter5",
    name: "Porter's Five Forces",
    description: "Analyze industry competitiveness.",
    fields: [
      { id: "threat_new", label: "Threat of New Entrants", type: "textarea", placeholder: "How easy is it for new competitors to enter?" },
      { id: "bargain_sup", label: "Bargaining Power of Suppliers", type: "textarea", placeholder: "How much control do suppliers have?" },
      { id: "bargain_buy", label: "Bargaining Power of Buyers", type: "textarea", placeholder: "How much control do customers have?" },
      { id: "threat_sub", label: "Threat of Substitutes", type: "textarea", placeholder: "What alternatives could replace your product?" },
      { id: "rivalry", label: "Industry Rivalry", type: "textarea", placeholder: "How intense is competition?" },
    ],
  },
  {
    key: "tam_sam_som",
    name: "TAM / SAM / SOM",
    description: "Market sizing overview.",
    fields: [
      { id: "tam", label: "TAM (Total Addressable Market)", type: "textarea", placeholder: "Size, assumptions, sources" },
      { id: "sam", label: "SAM (Serviceable Available Market)", type: "textarea", placeholder: "Portion you can realistically target" },
      { id: "som", label: "SOM (Serviceable Obtainable Market)", type: "textarea", placeholder: "Market share you can capture" },
      { id: "sources", label: "Sources / Citations", type: "textarea", placeholder: "Research sources and citations" },
    ],
  },
  {
    key: "pricing_table",
    name: "Pricing Table",
    description: "List tiers or SKUs with price & notes.",
    fields: [
      { id: "rows", label: "Rows (Name – Price – Notes)", type: "textarea", placeholder: "Basic – $19/mo – For starters\nPro – $49/mo – Most popular\nEnterprise – $199/mo – Custom SLAs" },
    ],
  },
];

export const STRUCTURED_TEMPLATES_V2 = V2_TEMPLATES;

export function renderV2TemplateMarkdown(key: string, data: Record<string,string>): string {
  if (key === "swot") {
    return `### SWOT Analysis

**Strengths**
${data.strengths || "-"}

**Weaknesses**
${data.weaknesses || "-"}

**Opportunities**
${data.opportunities || "-"}

**Threats**
${data.threats || "-"}`;
  }
  
  if (key === "persona") {
    return `### Customer Persona: ${data.name || "Unnamed"}

**Demographics**
${data.demographics || "-"}

**Goals / JTBD**
${data.jobs || "-"}

**Pain Points**
${data.pain || "-"}

**Acquisition Channels**
${data.channels || "-"}`;
  }

  if (key === "porter5") {
    return `### Porter's Five Forces

**Threat of New Entrants**
${data.threat_new||"-"}

**Bargaining Power of Suppliers**
${data.bargain_sup||"-"}

**Bargaining Power of Buyers**
${data.bargain_buy||"-"}

**Threat of Substitutes**
${data.threat_sub||"-"}

**Industry Rivalry**
${data.rivalry||"-"}`;
  }

  if (key === "tam_sam_som") {
    return `### Market Size: TAM / SAM / SOM

**TAM**
${data.tam||"-"}

**SAM**
${data.sam||"-"}

**SOM**
${data.som||"-"}

**Sources**
${data.sources||"-"}`;
  }

  if (key === "pricing_table") {
    const rows = (data.rows||"").split(/\n+/).filter(Boolean);
    const table = ["| Name | Price | Notes |", "|---|---:|---|"]
      .concat(rows.map((r)=>{ 
        const [name="", price="", notes=""] = r.split(" – "); 
        return `| ${name.trim()} | ${price.trim()} | ${notes.trim()} |`; 
      }))
      .join("\n");
    return `### Pricing

${table}`;
  }
  
  return "";
}

export function renderTemplateToMarkdownV2(key: string, data: Record<string, string>): string {
  return renderV2TemplateMarkdown(key, data);
}