import { initializeApp, getApps } from "firebase/app";
import {
  getAuth,
  signInAnonymously,
  onAuthStateChanged,
  User,
  GoogleAuthProvider,
  signInWithPopup,
  signOut,
  isSignInWithEmailLink,
  sendSignInLinkToEmail,
  signInWithEmailLink,
} from "firebase/auth";
import {
  getFirestore,
  doc,
  onSnapshot,
  setDoc,
  updateDoc,
  arrayUnion,
  arrayRemove,
} from "firebase/firestore";
import {
  getStorage,
  ref,
  uploadBytes,
  getDownloadURL,
} from "firebase/storage";

const cfg = {
  apiKey: import.meta.env.VITE_FIREBASE_API_KEY,
  authDomain: import.meta.env.VITE_FIREBASE_AUTH_DOMAIN || `${import.meta.env.VITE_FIREBASE_PROJECT_ID}.firebaseapp.com`,
  projectId: import.meta.env.VITE_FIREBASE_PROJECT_ID,
  appId: import.meta.env.VITE_FIREBASE_APP_ID,
  storageBucket: import.meta.env.VITE_FIREBASE_STORAGE_BUCKET || `${import.meta.env.VITE_FIREBASE_PROJECT_ID}.firebasestorage.app`,
  messagingSenderId: import.meta.env.VITE_FIREBASE_MESSAGING_SENDER_ID,
};

export function isFirebaseConfigured() {
  return !!(cfg.apiKey && cfg.authDomain && cfg.projectId && cfg.appId);
}

let _authReady: Promise<User | null> | null = null;

export function ensureFirebase() {
  if (!isFirebaseConfigured()) return null;
  
  // Use existing app if available, otherwise create new one
  let app;
  const existingApps = getApps();
  if (existingApps.length > 0) {
    app = existingApps[0];
  } else {
    app = initializeApp(cfg);
  }
  
  const auth = getAuth(app);
  const db = getFirestore(app);
  const storage = getStorage(app);

  // Complete pending magic link sign-in if URL contains it
  if (typeof window !== "undefined" && isSignInWithEmailLink(auth, window.location.href)) {
    const key = "ibrandbiz_emailForSignIn";
    let email = window.localStorage.getItem(key) || "";
    if (!email) {
      // As a last resort, prompt (tiny UX); you can replace with a modal if you prefer
      email = window.prompt("Confirm your email to complete sign in") || "";
    }
    if (email) {
      signInWithEmailLink(auth, email, window.location.href)
        .then(() => window.localStorage.removeItem(key))
        .catch(() => {/* ignore */});
    }
  }

  if (!_authReady) {
    _authReady = new Promise<User | null>((resolve) => {
      onAuthStateChanged(auth, async (u) => {
        if (u) return resolve(u);
        try {
          const cred = await signInAnonymously(auth);
          resolve(cred.user);
        } catch {
          resolve(null);
        }
      });
    });
  }

  return { app, auth, db, storage, authReady: _authReady };
}

// Legacy exports for backward compatibility - use same app instance
const legacyFirebase = ensureFirebase();
export const auth = legacyFirebase?.auth || getAuth();
export const db = legacyFirebase?.db || getFirestore();
export const storage = legacyFirebase?.storage || getStorage();
export default legacyFirebase?.app;

// ---- Sign-in helpers ----
export async function signInWithGoogle() {
  const fb = ensureFirebase();
  if (!fb) return null;
  const provider = new GoogleAuthProvider();
  const { auth } = fb;
  const res = await signInWithPopup(auth, provider);
  return res.user;
}

export async function startMagicLink(email: string) {
  const fb = ensureFirebase();
  if (!fb) return null;
  const { auth } = fb;
  const actionCodeSettings = {
    url: import.meta.env.VITE_APP_URL || window.location.origin,
    handleCodeInApp: true,
  };
  await sendSignInLinkToEmail(auth, email, actionCodeSettings);
  window.localStorage.setItem("ibrandbiz_emailForSignIn", email);
  return true;
}

export async function doSignOut() {
  const fb = ensureFirebase();
  if (!fb) return;
  await signOut(fb.auth);
}

// re-exports used by favorites hook and storage
export { getAuth, getFirestore, getStorage, doc, onSnapshot, setDoc, updateDoc, arrayUnion, arrayRemove, ref, uploadBytes, getDownloadURL };
