import type { FinanceInputs, FinanceOutputs, MonthlyRow } from "@shared/finance-types";

const MONTH_NAMES = ["Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"];

function formatMonthLabel(ym: string, i: number) {
  // ym = YYYY-MM
  const [y0, m0] = ym.split("-").map(Number);
  const base = new Date(y0, (m0 - 1) + i, 1);
  return `${MONTH_NAMES[base.getMonth()]} ${base.getFullYear()}`;
}

export function computePAndL(input: FinanceInputs): FinanceOutputs {
  const rows: MonthlyRow[] = [];
  let cumulative = 0;
  const price = Number(input.pricePerUnit) || 0;
  const cogsPct = Math.min(Math.max(input.cogsPercent, 0), 100) / 100;
  const varPerUnit = Number(input.variableExpensePerUnit || 0);
  const opex = Number(input.fixedMonthlyExpenses) || 0;

  for (let i = 0; i < 12; i++) {
    const units = Number(input.unitsPerMonth[i] || 0);
    const revenue = units * price;
    const cogs = revenue * cogsPct + units * varPerUnit;
    const grossProfit = revenue - cogs;
    const profit = grossProfit - opex;
    cumulative += profit;

    rows.push({
      month: formatMonthLabel(input.startMonth, i),
      revenue, cogs, grossProfit, opex, profit, cumulativeProfit: cumulative,
    });
  }

  const totalRevenue = rows.reduce((s, r) => s + r.revenue, 0);
  const totalProfit = rows.reduce((s, r) => s + r.profit, 0);

  // breakeven = first month where cumulativeProfit >= 0
  let breakevenMonthIndex: number | null = null;
  for (let i = 0; i < rows.length; i++) {
    if (rows[i].cumulativeProfit >= 0) { breakevenMonthIndex = i; break; }
  }

  let runwayMonths: number | null = null;
  if (input.startingCash != null) {
    let cash = input.startingCash;
    let m = 0;
    while (m < 60) { // cap at 5 years
      const idx = Math.min(m, rows.length - 1);
      cash += rows[idx].profit;
      if (cash <= 0) { runwayMonths = m + 1; break; }
      m++;
    }
    if (runwayMonths == null) runwayMonths = 60; // survival beyond horizon
  }

  return { rows, totalRevenue, totalProfit, breakevenMonthIndex, runwayMonths };
}