/**
 * PNG Export Utility
 * Converts SVG to PNG at specified sizes using offscreen canvas
 */

export async function svgToPng(svg: string, size: number): Promise<Blob> {
  return new Promise((resolve, reject) => {
    const canvas = document.createElement('canvas');
    canvas.width = size;
    canvas.height = size;
    
    const ctx = canvas.getContext('2d');
    if (!ctx) {
      reject(new Error('Failed to get canvas context'));
      return;
    }
    
    const img = new Image();
    const blob = new Blob([svg], { type: 'image/svg+xml;charset=utf-8' });
    const url = URL.createObjectURL(blob);
    
    img.onload = () => {
      // Clear canvas with transparency
      ctx.clearRect(0, 0, size, size);
      
      // Draw SVG maintaining aspect ratio and centering
      const aspectRatio = img.width / img.height;
      let drawWidth = size;
      let drawHeight = size;
      let offsetX = 0;
      let offsetY = 0;
      
      if (aspectRatio > 1) {
        // Wider than tall
        drawHeight = size / aspectRatio;
        offsetY = (size - drawHeight) / 2;
      } else if (aspectRatio < 1) {
        // Taller than wide
        drawWidth = size * aspectRatio;
        offsetX = (size - drawWidth) / 2;
      }
      
      ctx.drawImage(img, offsetX, offsetY, drawWidth, drawHeight);
      
      canvas.toBlob((blob) => {
        URL.revokeObjectURL(url);
        if (blob) {
          resolve(blob);
        } else {
          reject(new Error('Failed to create PNG blob'));
        }
      }, 'image/png');
    };
    
    img.onerror = () => {
      URL.revokeObjectURL(url);
      reject(new Error('Failed to load SVG image'));
    };
    
    img.src = url;
  });
}
