/**
 * PDF Export Utility
 * Converts SVG to PDF centered on Letter/A4 page using pdf-lib
 */

import { PDFDocument, rgb } from 'pdf-lib';

export async function svgToPdf(svg: string): Promise<Blob> {
  try {
    // Create PDF document (Letter size: 8.5" x 11" = 612 x 792 pts)
    const pdfDoc = await PDFDocument.create();
    const page = pdfDoc.addPage([612, 792]);
    
    // For browser compatibility, we'll embed a PNG version
    // (true vector SVG embedding requires server-side or complex libraries)
    const img = new Image();
    const blob = new Blob([svg], { type: 'image/svg+xml;charset=utf-8' });
    const url = URL.createObjectURL(blob);
    
    // Capture the original aspect ratio and create canvas to match
    let originalAspectRatio = 1;
    
    const pngData = await new Promise<Uint8Array>((resolve, reject) => {
      img.onload = () => {
        // Store the aspect ratio from natural dimensions
        originalAspectRatio = img.width / img.height;
        
        // Create canvas with dimensions matching the aspect ratio (not square)
        const baseSize = 1024;
        let canvasWidth = baseSize;
        let canvasHeight = baseSize;
        
        if (originalAspectRatio > 1) {
          // Wider than tall - full width, reduced height
          canvasHeight = baseSize / originalAspectRatio;
        } else if (originalAspectRatio < 1) {
          // Taller than wide - full height, reduced width
          canvasWidth = baseSize * originalAspectRatio;
        }
        
        const canvas = document.createElement('canvas');
        canvas.width = canvasWidth;
        canvas.height = canvasHeight;
        
        const ctx = canvas.getContext('2d');
        if (!ctx) {
          reject(new Error('Failed to get canvas context'));
          return;
        }
        
        // Draw image filling the entire canvas (no letterboxing needed)
        ctx.clearRect(0, 0, canvasWidth, canvasHeight);
        ctx.drawImage(img, 0, 0, canvasWidth, canvasHeight);
        
        canvas.toBlob(async (pngBlob) => {
          URL.revokeObjectURL(url);
          if (pngBlob) {
            const arrayBuffer = await pngBlob.arrayBuffer();
            resolve(new Uint8Array(arrayBuffer));
          } else {
            reject(new Error('Failed to create PNG'));
          }
        }, 'image/png');
      };
      
      img.onerror = () => {
        URL.revokeObjectURL(url);
        reject(new Error('Failed to load SVG'));
      };
      
      img.src = url;
    });
    
    // Embed PNG in PDF
    const pngImage = await pdfDoc.embedPng(pngData);
    
    // Calculate proper dimensions based on original aspect ratio, not square canvas
    const maxDimension = 512; // Max size on page (in points)
    let pdfWidth = maxDimension;
    let pdfHeight = maxDimension;
    
    if (originalAspectRatio > 1) {
      // Wider than tall - constrain width, calculate height
      pdfHeight = maxDimension / originalAspectRatio;
    } else if (originalAspectRatio < 1) {
      // Taller than wide - constrain height, calculate width
      pdfWidth = maxDimension * originalAspectRatio;
    }
    
    // Center on page
    const x = (page.getWidth() - pdfWidth) / 2;
    const y = (page.getHeight() - pdfHeight) / 2;
    
    page.drawImage(pngImage, {
      x,
      y,
      width: pdfWidth,
      height: pdfHeight,
    });
    
    const pdfBytes = await pdfDoc.save();
    return new Blob([pdfBytes], { type: 'application/pdf' });
  } catch (error) {
    console.error('PDF generation error:', error);
    throw new Error('Failed to generate PDF');
  }
}
