/**
 * IBrandBiz — Consent Utilities
 * Based on SuperNova's specification
 * 
 * Provides type-safe cookie consent management with localStorage persistence.
 * Essential cookies are always enabled, analytics and marketing are user-controlled.
 */

/* ========================= Consent Types ========================= */

export type ConsentState = {
  essential: true; // always true, non-optional
  analytics: boolean;
  marketing: boolean;
};

/* ========================= Constants ========================= */

const CONSENT_KEY = "ibrandbiz.cookieConsent.v1";

/* ========================= Consent Helpers ====================== */

/**
 * Retrieve the current consent state from localStorage
 * @returns ConsentState if consent exists, null if no consent or parsing fails
 */
export function getConsent(): ConsentState | null {
  try {
    const raw = localStorage.getItem(CONSENT_KEY);
    return raw ? (JSON.parse(raw) as ConsentState) : null;
  } catch {
    return null;
  }
}

/**
 * Store consent state in localStorage
 * @param next - The consent state to store
 */
export function setConsent(next: ConsentState): void {
  localStorage.setItem(CONSENT_KEY, JSON.stringify(next));
}

/**
 * Check if a specific consent category is enabled
 * @param category - The consent category to check (essential, analytics, marketing)
 * @returns true if category is consented, false if not consented or no consent exists
 */
export function hasConsentCategory(category: keyof ConsentState): boolean {
  const c = getConsent();
  if (!c) return false;
  return !!c[category];
}