// Icon layer helpers for Logo Creator
// Manages adding, replacing, and positioning AI-generated icon SVGs

export interface IconLayer {
  id: string;
  type: 'icon';
  svg: string;
  transform: {
    x: number; // percentage
    y: number; // percentage
    scale: number; // multiplier
    rotation: number; // degrees
  };
}

/**
 * Create a new icon layer from SVG string
 * Centers it and scales to 60% of artboard
 */
export function createIconLayer(svg: string, layerName?: string): IconLayer {
  const id = layerName || `icon-${Date.now()}`;
  
  return {
    id,
    type: 'icon',
    svg,
    transform: {
      x: 50, // center horizontally
      y: 50, // center vertically
      scale: 0.6, // 60% of artboard
      rotation: 0
    }
  };
}

/**
 * Replace SVG content in existing layer while preserving transform
 */
export function replaceIconLayerSvg(layer: IconLayer, newSvg: string): IconLayer {
  return {
    ...layer,
    svg: newSvg
    // Preserve existing transform
  };
}

/**
 * Generate incremental layer name (Icon 2, Icon 3, etc.)
 */
export function generateLayerName(existingNames: string[]): string {
  let counter = 1;
  let name = 'Icon';
  
  while (existingNames.includes(name)) {
    counter++;
    name = `Icon ${counter}`;
  }
  
  return name;
}

/**
 * Fit icon to artboard dimensions while maintaining aspect ratio
 * Returns adjusted scale factor
 */
export function fitToArtboard(
  iconWidth: number,
  iconHeight: number,
  artboardWidth: number,
  artboardHeight: number,
  maxScale: number = 0.8 // Don't exceed 80% of artboard
): number {
  const widthRatio = artboardWidth / iconWidth;
  const heightRatio = artboardHeight / iconHeight;
  
  // Use smaller ratio to ensure it fits
  const fitScale = Math.min(widthRatio, heightRatio, maxScale);
  
  return fitScale;
}

/**
 * Extract viewBox dimensions from SVG string
 */
export function getIconDimensions(svg: string): { width: number; height: number } | null {
  try {
    const parser = new DOMParser();
    const doc = parser.parseFromString(svg, 'image/svg+xml');
    const svgEl = doc.querySelector('svg');
    
    if (!svgEl) return null;
    
    const viewBox = svgEl.getAttribute('viewBox');
    if (viewBox) {
      const [, , width, height] = viewBox.split(/\s+/).map(Number);
      return { width, height };
    }
    
    // Fallback to width/height attributes
    const width = parseFloat(svgEl.getAttribute('width') || '100');
    const height = parseFloat(svgEl.getAttribute('height') || '100');
    return { width, height };
    
  } catch {
    return null;
  }
}
