// src/lib/brand.ts
import type { BrandKit } from "@shared/brand-types";

export function getBrandPalette(brand: BrandKit){
  const p = brand.primary || "#6D28D9";
  const s = brand.secondary || "#82ca9d";
  const a = brand.accent || "#ffc658";
  return [p, s, a, shadeHex(p, -12), shadeHex(s, -12), shadeHex(a, -12)];
}

export function shadeHex(hex: string, percent: number){
  try{
    const h = hex.replace("#","");
    const num = parseInt(h,16);
    const r = Math.min(255, Math.max(0, (num>>16) + Math.round(2.55*percent)));
    const g = Math.min(255, Math.max(0, ((num>>8)&0xFF) + Math.round(2.55*percent)));
    const b = Math.min(255, Math.max(0, (num&0xFF) + Math.round(2.55*percent)));
    return `#${(1<<24 | (r<<16) | (g<<8) | b).toString(16).slice(1)}`;
  } catch { return hex; }
}

/**
 * Convert hex color to HSL
 */
export function hexToHsl(hex: string): { h: number; s: number; l: number } {
  try {
    const r = parseInt(hex.slice(1, 3), 16) / 255;
    const g = parseInt(hex.slice(3, 5), 16) / 255;
    const b = parseInt(hex.slice(5, 7), 16) / 255;

    const max = Math.max(r, g, b);
    const min = Math.min(r, g, b);
    const diff = max - min;
    const sum = max + min;
    const l = sum / 2;

    let h = 0;
    let s = 0;

    if (diff !== 0) {
      s = l > 0.5 ? diff / (2 - sum) : diff / sum;

      switch (max) {
        case r:
          h = ((g - b) / diff) + (g < b ? 6 : 0);
          break;
        case g:
          h = (b - r) / diff + 2;
          break;
        case b:
          h = (r - g) / diff + 4;
          break;
      }
      h /= 6;
    }

    return {
      h: Math.round(h * 360),
      s: Math.round(s * 100),
      l: Math.round(l * 100)
    };
  } catch {
    return { h: 0, s: 0, l: 50 };
  }
}

/**
 * Check if a color is light or dark (for text contrast)
 */
export function isLightColor(hex: string): boolean {
  try {
    const { l } = hexToHsl(hex);
    return l > 50;
  } catch {
    return true;
  }
}

/**
 * Get appropriate text color (black or white) for a background color
 */
export function getContrastTextColor(backgroundColor: string): string {
  return isLightColor(backgroundColor) ? "#000000" : "#ffffff";
}

/**
 * Apply brand kit styles as CSS custom properties
 */
export function applyBrandKitStyles(brand: BrandKit): void {
  const root = document.documentElement;
  
  root.style.setProperty("--brand-primary", brand.primary);
  root.style.setProperty("--brand-secondary", brand.secondary || "#06B6D4");
  root.style.setProperty("--brand-accent", brand.accent || "#F59E0B");
  root.style.setProperty("--brand-font-head", brand.fontHead || "Inter, ui-sans-serif");
  root.style.setProperty("--brand-font-body", brand.fontBody || "Inter, ui-sans-serif");
  
  // Apply palette colors
  const palette = getBrandPalette(brand);
  palette.forEach((color, index) => {
    root.style.setProperty(`--brand-color-${index}`, color);
  });
}

/**
 * Generate a gradient string from brand colors
 */
export function getBrandGradient(brand: BrandKit, direction = "135deg"): string {
  const primary = brand.primary || "#6D28D9";
  const accent = brand.accent || brand.secondary || "#F59E0B";
  return `linear-gradient(${direction}, ${primary}, ${accent})`;
}