import path from "path";
import fs from "fs";

export type BPItem = {
  id: string;
  name: string;
  category?: string;
  tags?: string[];
  previewUrl: string;
  docxUrl: string;
  updatedAt?: string;
  sections?: string[];
};

export type BPManifest = {
  collection: "business-plan";
  version: number;
  items: BPItem[];
};

export const PUBLIC_DIR = path.join(process.cwd(), "public");
export const DOC_DIR = path.join(PUBLIC_DIR, "templates", "business-plan", "docs");
export const PREV_DIR = path.join(PUBLIC_DIR, "templates", "business-plan", "previews");
export const DATA_DIR = path.join(PUBLIC_DIR, "site", "data");
export const MANIFEST = path.join(DATA_DIR, "manifest.bp.json");

export function ensureDirs() {
  [DOC_DIR, PREV_DIR, DATA_DIR].forEach(d => { if (!fs.existsSync(d)) fs.mkdirSync(d, { recursive: true }); });
}

export function loadManifest(): BPManifest {
  ensureDirs();
  if (!fs.existsSync(MANIFEST)) {
    const fresh: BPManifest = { collection: "business-plan", version: 1, items: [] };
    fs.writeFileSync(MANIFEST, JSON.stringify(fresh, null, 2), "utf8");
    return fresh;
  }
  const raw = fs.readFileSync(MANIFEST, "utf8");
  try {
    const json = JSON.parse(raw);
    json.collection = "business-plan";
    json.version = Number(json.version || 1);
    json.items = Array.isArray(json.items) ? json.items : [];
    return json as BPManifest;
  } catch {
    return { collection: "business-plan", version: 1, items: [] };
  }
}

export function saveManifest(m: BPManifest) {
  m.collection = "business-plan";
  m.version = Number(m.version || 1);
  fs.writeFileSync(MANIFEST, JSON.stringify(m, null, 2), "utf8");
}

export const kebab = (s: string) =>
  (s || "").toLowerCase().trim().replace(/[^a-z0-9\-]/g, "-").replace(/\-+/g, "-");

export function upsert(items: BPItem[], item: BPItem): BPItem[] {
  const idx = items.findIndex(x => x.id === item.id);
  if (idx >= 0) items[idx] = item; else items.push(item);
  return items;
}