import { useState, useEffect } from "react";
import { useLogo } from "@/hooks/useLogo";
import { Link } from "wouter";
import { Button } from "@/components/ui/button";
import { ChevronUp, Menu, X } from "lucide-react";
import Footer from "@/components/Footer";

// Import reusable sections
import Hero from "@/marketing/sections/Hero";
import Services from "@/marketing/sections/Services";
import Products from "@/marketing/sections/Products";
import Pricing from "@/marketing/sections/Pricing";
import FAQ from "@/marketing/sections/FAQ";
import Contact from "@/marketing/sections/Contact";

function FloatingMobileNav() {
  const [activeSection, setActiveSection] = useState("");

  const items = [
    { id: "hero", label: "Home" },
    { id: "services", label: "Services" },
    { id: "products", label: "Products" },
    { id: "pricing", label: "Pricing" },
    { id: "faq", label: "FAQ" },
    { id: "contact", label: "Contact" },
  ];

  // IntersectionObserver to track active section
  useEffect(() => {
    const observer = new IntersectionObserver(
      (entries) => {
        entries.forEach((entry) => {
          if (entry.isIntersecting) {
            setActiveSection(entry.target.id);
          }
        });
      },
      {
        threshold: 0.3,
        rootMargin: "-20% 0px -70% 0px",
      }
    );

    items.forEach((item) => {
      const element = document.getElementById(item.id);
      if (element) observer.observe(element);
    });

    return () => observer.disconnect();
  }, []);

  const scrollToSection = (sectionId: string) => {
    const element = document.getElementById(sectionId);
    if (element) {
      element.scrollIntoView({ behavior: "smooth" });
    }
  };

  return (
    <nav className="fixed bottom-4 left-1/2 -translate-x-1/2 z-50 bg-white/90 dark:bg-gray-900/90 backdrop-blur rounded-full shadow-lg px-3 py-2 flex gap-2 overflow-x-auto max-w-[95vw] border">
      {items.map((item) => (
        <button
          key={item.id}
          onClick={() => scrollToSection(item.id)}
          className={`text-xs px-3 py-1 rounded-full border transition-all whitespace-nowrap ${
            activeSection === item.id
              ? "bg-primary text-white border-primary"
              : "hover:bg-gray-100 dark:hover:bg-gray-800 border-gray-200 dark:border-gray-700"
          }`}
        >
          {item.label}
        </button>
      ))}
    </nav>
  );
}

export default function MobileMarketingLayout() {
  const [showNav, setShowNav] = useState(false);
  const logoSrc = useLogo();

  const scrollToTop = () => {
    window.scrollTo({ top: 0, behavior: "smooth" });
  };

  return (
    <div className="mobile-marketing min-h-screen bg-background">
      {/* Fixed Mobile Header */}
      <header className="fixed top-0 left-0 right-0 z-40 bg-white/95 dark:bg-gray-900/95 backdrop-blur-sm border-b shadow-sm">
        <div className="flex items-center justify-between p-4">
          <button onClick={scrollToTop} className="flex items-center">
            <img 
              src={logoSrc} 
              alt="IBrandBiz Logo" 
              className="h-12 w-auto"
            />
          </button>
          <Button
            variant="ghost"
            size="sm"
            onClick={() => setShowNav(!showNav)}
            className="p-2"
          >
            {showNav ? <X className="h-5 w-5" /> : <Menu className="h-5 w-5" />}
          </Button>
        </div>
        
        {/* Mobile Navigation Dropdown */}
        {showNav && (
          <div className="absolute top-full left-0 right-0 bg-white dark:bg-gray-900 border-t shadow-lg">
            <nav className="p-4 space-y-3">
              <Link href="/about">
                <button className="block w-full text-left py-2 px-3 text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-800 rounded">
                  About
                </button>
              </Link>
              <Link href="/resources">
                <button className="block w-full text-left py-2 px-3 text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-800 rounded">
                  Resources
                </button>
              </Link>
              <Link href="/terms">
                <button className="block w-full text-left py-2 px-3 text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-800 rounded">
                  Terms
                </button>
              </Link>
              <Link href="/privacy">
                <button className="block w-full text-left py-2 px-3 text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-800 rounded">
                  Privacy
                </button>
              </Link>
              <hr className="my-2" />
              <Link href="/signup">
                <Button className="w-full bg-primary hover:bg-primary/90 text-white" onClick={() => setShowNav(false)}>
                  Get Started
                </Button>
              </Link>
            </nav>
          </div>
        )}
      </header>

      {/* Main Content - Long Scroll */}
      <main className="pt-20"> {/* Account for fixed header */}
        <FloatingMobileNav />
        <Hero />
        <Services />
        <Products />
        <Pricing />
        <FAQ />
        <Contact />
      </main>

      {/* Footer */}
      <Footer />

      {/* Floating Back to Top Button */}
      <Button
        onClick={scrollToTop}
        className="fixed bottom-20 right-6 z-30 w-12 h-12 rounded-full shadow-lg bg-primary hover:bg-primary/90 text-white"
        size="sm"
      >
        <ChevronUp className="h-5 w-5" />
      </Button>
    </div>
  );
}