/**
 * React useRole Hook - Client-side role-based conditional rendering
 * 
 * Provides role checking and conditional rendering based on Firebase custom claims
 * Automatically refreshes tokens to get latest role claims
 */

import React, { useState, useEffect, useCallback } from 'react';
import { useAuth } from '@/contexts/AuthContext';
import type { Role } from '@shared/schema';

// Role hierarchy matching backend (owner(4) > management(3) > creator(2.5) > staff(2) > analyst(1) > pro/user(0))
const ROLE_HIERARCHY: Record<Role, number> = {
  'owner': 4,
  'management': 3,
  'manager': 3, // Compatibility alias for management
  'creator': 2.5,
  'staff': 2,
  'analyst': 1,
  'pro': 0,
  'user': 0
};

const ADMIN_ROLES: Role[] = ['owner', 'management', 'manager', 'staff', 'analyst'];
const CREATOR_DASHBOARD_ROLES: Role[] = ['creator', 'owner', 'management', 'manager', 'staff', 'analyst'];

interface UseRoleReturn {
  role: Role | null;
  isLoading: boolean;
  hasRole: (requiredRole: Role) => boolean;
  hasAnyRole: (roles: Role[]) => boolean;
  isAdmin: boolean;
  isOwner: boolean;
  isManagement: boolean;
  isStaff: boolean;
  isAnalyst: boolean;
  isPro: boolean;
  refreshRole: () => Promise<void>;
  roleLevel: number;
}

export function useRole(): UseRoleReturn {
  const { user } = useAuth();
  const [role, setRole] = useState<Role | null>(null);
  const [isLoading, setIsLoading] = useState(true);

  // Extract role from Firebase custom claims
  const extractRoleFromToken = useCallback(async (): Promise<Role | null> => {
    if (!user) return null;
    
    try {
      // Force token refresh to get latest claims
      const idTokenResult = await user.getIdTokenResult(true);
      const claims = idTokenResult.claims;
      
      console.log('🔍 Firebase claims:', claims);
      
      // Check for role-based claim first
      if (claims.role && typeof claims.role === 'string') {
        const userRole = claims.role as Role;
        console.log(`✅ Role claim found: ${userRole}`);
        return userRole;
      }
      
      // Check for legacy admin claim (backward compatibility)
      if (claims.admin === true) {
        console.log('⚠️ Legacy admin claim detected, treating as owner');
        return 'owner';
      }
      
      // Default to user role if no claims
      console.log('ℹ️ No role claims found, defaulting to user');
      return 'user';
    } catch (error) {
      console.error('❌ Failed to extract role from token:', error);
      return 'user'; // Fallback to user role
    }
  }, [user]);

  // Refresh role from Firebase
  const refreshRole = useCallback(async () => {
    setIsLoading(true);
    try {
      const extractedRole = await extractRoleFromToken();
      setRole(extractedRole);
      console.log(`🔄 Role refreshed: ${extractedRole}`);
    } catch (error) {
      console.error('Failed to refresh role:', error);
      setRole('user'); // Fallback
    } finally {
      setIsLoading(false);
    }
  }, [extractRoleFromToken]);

  // Initial role extraction and automatic refresh on auth state change
  useEffect(() => {
    if (user) {
      refreshRole();
    } else {
      setRole(null);
      setIsLoading(false);
    }
  }, [user, refreshRole]);

  // Check if user has at least the required role level
  const hasRole = useCallback((requiredRole: Role): boolean => {
    if (!role) return false;
    return ROLE_HIERARCHY[role] >= ROLE_HIERARCHY[requiredRole];
  }, [role]);

  // Check if user has any of the specified roles
  const hasAnyRole = useCallback((roles: Role[]): boolean => {
    if (!role) return false;
    return roles.some(requiredRole => hasRole(requiredRole));
  }, [role, hasRole]);

  // Computed properties
  const isAdmin = role ? ADMIN_ROLES.includes(role) : false;
  const isOwner = role === 'owner';
  const isManagement = hasRole('management');
  const isStaff = hasRole('staff');
  const isAnalyst = hasRole('analyst');
  const isPro = role === 'pro';
  const roleLevel = role ? ROLE_HIERARCHY[role] : 0;

  return {
    role,
    isLoading,
    hasRole,
    hasAnyRole,
    isAdmin,
    isOwner,
    isManagement,
    isStaff,
    isAnalyst,
    isPro,
    refreshRole,
    roleLevel
  };
}

/**
 * Higher-order component for role-based conditional rendering
 */
interface RoleGuardProps {
  requiredRole?: Role;
  requiredRoles?: Role[];
  fallback?: React.ReactNode;
  children: React.ReactNode;
}

export function RoleGuard({ 
  requiredRole, 
  requiredRoles, 
  fallback = null, 
  children 
}: RoleGuardProps) {
  const { hasRole, hasAnyRole, isLoading } = useRole();

  if (isLoading) {
    return React.createElement('div', { className: "animate-pulse" }, 'Loading permissions...');
  }

  const hasPermission = requiredRole ? hasRole(requiredRole) : 
                       requiredRoles ? hasAnyRole(requiredRoles) : 
                       true;

  return hasPermission ? React.createElement(React.Fragment, null, children) : React.createElement(React.Fragment, null, fallback);
}

/**
 * Component for admin-only content
 */
interface AdminOnlyProps {
  fallback?: React.ReactNode;
  children: React.ReactNode;
}

export function AdminOnly({ fallback = null, children }: AdminOnlyProps) {
  return (
    <RoleGuard requiredRoles={ADMIN_ROLES} fallback={fallback}>
      {children}
    </RoleGuard>
  );
}

/**
 * Hook for checking specific role permissions
 */
export function useRoleCheck() {
  const { hasRole, hasAnyRole, isAdmin } = useRole();
  
  return {
    canManageUsers: hasRole('management'),
    canViewAnalytics: hasRole('analyst'),
    canManageContent: hasRole('staff'),
    canAccessAdmin: isAdmin,
    hasRole,
    hasAnyRole
  };
}