import { useState } from 'react';
import { useAuth } from '@/contexts/AuthContext';
import { getUserPaidStatus } from '@/services/authRedirect';
import { useToast } from '@/hooks/use-toast';
import { AuthModal } from '@/components/AuthModal';
import { PaywallModal } from '@/components/PaywallModal';

interface UseRequireProOptions {
  feature?: string;
  description?: string;
}

/**
 * Hook for gating Pro features at the action level (e.g., exports, saves)
 * Returns a guard function that checks auth/payment status before executing an action
 */
export function useRequirePro(options: UseRequireProOptions = {}) {
  const { currentUser, loading } = useAuth();
  const isPaid = getUserPaidStatus();
  const { toast } = useToast();
  
  const [showAuthModal, setShowAuthModal] = useState(false);
  const [showPaywallModal, setShowPaywallModal] = useState(false);

  /**
   * Guards an action, checking if user is authenticated and has Pro subscription
   * @param action The action to execute if user has access
   * @param actionName Optional name for the action (for analytics/logging)
   */
  const requirePro = async (action: () => void | Promise<void>, actionName?: string) => {
    // Don't execute during loading
    if (loading) {
      toast({
        title: "Please wait",
        description: "Loading user information...",
      });
      return;
    }

    // Gate unauthenticated users
    if (!currentUser) {
      console.log(`Gated unauthenticated user from action: ${actionName || 'unknown'}`);
      setShowAuthModal(true);
      return;
    }

    // Gate non-paid users
    if (!isPaid) {
      console.log(`Gated non-paid user from action: ${actionName || 'unknown'}`);
      setShowPaywallModal(true);
      return;
    }

    // Execute the action for paid users
    try {
      await action();
      console.log(`Executed Pro action: ${actionName || 'unknown'}`);
    } catch (error) {
      console.error(`Failed to execute Pro action "${actionName}":`, error);
      toast({
        title: "Action failed",
        description: "Please try again.",
        variant: "destructive",
      });
    }
  };

  /**
   * Simple boolean check for conditional UI rendering
   */
  const hasProAccess = currentUser && isPaid;

  /**
   * Simple boolean check for showing loading states
   */
  const isCheckingAccess = loading;

  const authModal = showAuthModal && (
    <AuthModal
      isOpen={true}
      onClose={() => setShowAuthModal(false)}
    />
  );

  const paywallModal = showPaywallModal && (
    <PaywallModal
      isOpen={true}
      onClose={() => setShowPaywallModal(false)}
      feature={options.feature || "premium features"}
      description={options.description || "Access advanced export and save functionality"}
    />
  );

  return {
    requirePro,
    hasProAccess,
    isCheckingAccess,
    authModal,
    paywallModal,
  };
}