import { useEffect, useState } from "react";
import { LogoManifestSchema, LogoTemplateManifest, LogoTemplateItem } from "../types/logo-manifest";

type State =
  | { status: "idle" | "loading" }
  | { status: "ready"; manifest: LogoTemplateManifest; items: LogoTemplateItem[] }
  | { status: "error"; error: string };

export function useLogoManifest(path = "/site/data/manifest.logo.json"): State {
  const [state, setState] = useState<State>({ status: "idle" });

  useEffect(() => {
    let cancelled = false;
    async function run() {
      try {
        setState({ status: "loading" });
        const res = await fetch(path, { cache: "no-cache" });
        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        const json = await res.json();
        const parsed = LogoManifestSchema.safeParse(json);
        if (!parsed.success) {
          throw new Error("Manifest validation failed: " + parsed.error.message);
        }
        if (!cancelled) {
          setState({ status: "ready", manifest: parsed.data, items: parsed.data.items });
        }
      } catch (e: any) {
        if (!cancelled) setState({ status: "error", error: e?.message || "Unknown error" });
      }
    }
    run();
    return () => { cancelled = true; };
  }, [path]);

  return state;
}