import { createContext, useContext, useReducer, useEffect, ReactNode } from 'react';
import { useAuth } from '@/contexts/AuthContext';
import { toast } from 'sonner';
import { v4 as uuidv4 } from 'uuid';

export interface CartItem {
  id: string;
  itemType: 'stock_photo' | 'subscription' | 'creator_asset';
  itemId: string;
  itemName: string;
  itemPrice: number; // Price in cents
  quantity: number;
  metadata?: Record<string, any>;
}

interface CartState {
  items: CartItem[];
  sessionId: string;
  isLoading: boolean;
}

type CartAction = 
  | { type: 'ADD_ITEM'; item: CartItem }
  | { type: 'REMOVE_ITEM'; id: string }
  | { type: 'UPDATE_QUANTITY'; id: string; quantity: number }
  | { type: 'CLEAR_CART' }
  | { type: 'LOAD_CART'; items: CartItem[] }
  | { type: 'SET_LOADING'; loading: boolean };

interface CartContextValue {
  items: CartItem[];
  itemCount: number;
  totalCents: number;
  sessionId: string;
  isLoading: boolean;
  addItem: (item: Omit<CartItem, 'id'>, options?: { forceReplace?: boolean }) => boolean;
  removeItem: (id: string) => void;
  updateQuantity: (id: string, quantity: number) => void;
  clearCart: () => void;
  hasItem: (itemId: string, itemType: string) => boolean;
  canAddItem: (item: Omit<CartItem, 'id'>) => { canAdd: boolean; reason?: string; conflictCreator?: string };
}

const CartContext = createContext<CartContextValue | undefined>(undefined);

function cartReducer(state: CartState, action: CartAction): CartState {
  switch (action.type) {
    case 'ADD_ITEM': {
      // Check if item already exists
      const existingItemIndex = state.items.findIndex(
        item => item.itemId === action.item.itemId && item.itemType === action.item.itemType
      );

      if (existingItemIndex >= 0) {
        // For creator assets, do NOT merge quantities - digital licenses are limited to 1
        if (action.item.itemType === 'creator_asset') {
          // Don't add duplicate creator assets - they should be blocked at validation level
          console.warn('Attempted to add duplicate creator asset - ignoring');
          return state;
        }
        
        // Update quantity of existing item for non-creator assets
        const updatedItems = [...state.items];
        updatedItems[existingItemIndex].quantity += action.item.quantity;
        return { ...state, items: updatedItems };
      }

      // Add new item
      return {
        ...state,
        items: [...state.items, action.item]
      };
    }
    case 'REMOVE_ITEM':
      return {
        ...state,
        items: state.items.filter(item => item.id !== action.id)
      };
    case 'UPDATE_QUANTITY':
      return {
        ...state,
        items: state.items.map(item => {
          if (item.id === action.id) {
            // For creator assets, always clamp quantity to 1 (digital license constraint)
            if (item.itemType === 'creator_asset') {
              return { ...item, quantity: 1 };
            }
            return { ...item, quantity: action.quantity };
          }
          return item;
        })
      };
    case 'CLEAR_CART':
      return {
        ...state,
        items: []
      };
    case 'LOAD_CART':
      return {
        ...state,
        items: action.items
      };
    case 'SET_LOADING':
      return {
        ...state,
        isLoading: action.loading
      };
    default:
      return state;
  }
}

function generateSessionId(): string {
  let sessionId = localStorage.getItem('cart_session_id');
  if (!sessionId) {
    sessionId = uuidv4();
    localStorage.setItem('cart_session_id', sessionId);
  }
  return sessionId;
}

export function CartProvider({ children }: { children: ReactNode }) {
  const { user } = useAuth();
  const [state, dispatch] = useReducer(cartReducer, {
    items: [],
    sessionId: generateSessionId(),
    isLoading: false
  });

  // Load cart from localStorage on mount
  useEffect(() => {
    const savedItems = localStorage.getItem('cart_items');
    if (savedItems) {
      try {
        const items = JSON.parse(savedItems);
        dispatch({ type: 'LOAD_CART', items });
      } catch (error) {
        console.error('Failed to parse cart items from localStorage:', error);
      }
    }
  }, []);

  // Save cart to localStorage whenever items change
  useEffect(() => {
    localStorage.setItem('cart_items', JSON.stringify(state.items));
  }, [state.items]);

  // Clear cart when user logs out
  useEffect(() => {
    if (user === null) {
      // User logged out, clear cart
      dispatch({ type: 'CLEAR_CART' });
    }
  }, [user]);

  const canAddItem = (item: Omit<CartItem, 'id'>) => {
    const existingItems = state.items;
    
    // For creator assets, enforce single-seller constraint and quantity limits
    if (item.itemType === 'creator_asset') {
      const existingCreatorAssets = existingItems.filter(cartItem => cartItem.itemType === 'creator_asset');
      const existingStockPhotos = existingItems.filter(cartItem => cartItem.itemType === 'stock_photo');
      const existingSubscriptions = existingItems.filter(cartItem => cartItem.itemType === 'subscription');
      
      // Check if item already exists (should be blocked - digital licenses can't have quantity > 1)
      const existingItem = existingItems.find(
        cartItem => cartItem.itemId === item.itemId && cartItem.itemType === item.itemType
      );
      if (existingItem) {
        return { canAdd: false, reason: 'Digital license already in cart. Only one copy per asset allowed.' };
      }

      // Prevent mixing creator assets with stock photos
      if (existingStockPhotos.length > 0) {
        return { 
          canAdd: false, 
          reason: 'Creator assets cannot be mixed with stock photos. Please purchase separately.',
          conflictCreator: 'Stock Photos'
        };
      }

      // Prevent mixing creator assets with subscriptions
      if (existingSubscriptions.length > 0) {
        return { 
          canAdd: false, 
          reason: 'Creator assets cannot be mixed with subscription plans. Please purchase separately.',
          conflictCreator: 'Subscription Plan'
        };
      }

      // Check for different creator (single-seller constraint)
      if (existingCreatorAssets.length > 0) {
        const existingCreatorId = existingCreatorAssets[0].metadata?.creatorId;
        const newCreatorId = item.metadata?.creatorId;
        
        if (existingCreatorId && newCreatorId && existingCreatorId !== newCreatorId) {
          const conflictCreator = existingItems.find(cartItem => 
            cartItem.itemType === 'creator_asset' && cartItem.metadata?.creatorId === existingCreatorId
          );
          return { 
            canAdd: false, 
            reason: 'Cart can only contain assets from one creator at a time.',
            conflictCreator: conflictCreator?.metadata?.creatorName || 'Unknown Creator'
          };
        }
      }

      // Enforce quantity = 1 for digital licenses
      if (item.quantity > 1) {
        return { canAdd: false, reason: 'Digital licenses are limited to one copy per asset.' };
      }
    }

    // For stock photos, prevent mixing with creator assets
    if (item.itemType === 'stock_photo') {
      const existingCreatorAssets = existingItems.filter(cartItem => cartItem.itemType === 'creator_asset');
      if (existingCreatorAssets.length > 0) {
        const conflictCreator = existingCreatorAssets[0].metadata?.creatorName;
        return { 
          canAdd: false, 
          reason: 'Stock photos cannot be mixed with creator assets. Please purchase separately.',
          conflictCreator: conflictCreator || 'Creator Assets'
        };
      }
    }

    // For subscriptions, prevent mixing with creator assets
    if (item.itemType === 'subscription') {
      const existingCreatorAssets = existingItems.filter(cartItem => cartItem.itemType === 'creator_asset');
      if (existingCreatorAssets.length > 0) {
        const conflictCreator = existingCreatorAssets[0].metadata?.creatorName;
        return { 
          canAdd: false, 
          reason: 'Subscription plans cannot be mixed with creator assets. Please purchase separately.',
          conflictCreator: conflictCreator || 'Creator Assets'
        };
      }
    }

    return { canAdd: true };
  };

  const addItem = (item: Omit<CartItem, 'id'>, options?: { forceReplace?: boolean }) => {
    // Check constraints first
    const validation = canAddItem(item);
    
    if (!validation.canAdd && !options?.forceReplace) {
      // Show appropriate toast message
      if (validation.reason?.includes('one creator')) {
        toast.error(validation.reason, {
          action: {
            label: 'Replace Cart',
            onClick: () => {
              clearCart();
              addItem(item, { forceReplace: true });
              toast.success('Cart replaced with new item');
            }
          },
          duration: 5000
        });
      } else {
        toast.error(validation.reason);
      }
      return false;
    }

    // For creator assets, always set quantity to 1 (digital license constraint)
    const finalItem = item.itemType === 'creator_asset' 
      ? { ...item, quantity: 1 }
      : item;

    const newItem: CartItem = {
      ...finalItem,
      id: uuidv4()
    };
    
    dispatch({ type: 'ADD_ITEM', item: newItem });
    return true;
  };

  const removeItem = (id: string) => {
    dispatch({ type: 'REMOVE_ITEM', id });
  };

  const updateQuantity = (id: string, quantity: number) => {
    if (quantity <= 0) {
      removeItem(id);
      return;
    }
    
    // Find the item to check its type
    const item = state.items.find(cartItem => cartItem.id === id);
    if (item && item.itemType === 'creator_asset') {
      // For creator assets, show toast feedback when attempting to change quantity
      if (quantity !== 1) {
        toast.error('Digital licenses are limited to one copy per asset. Quantity cannot be changed.');
        return; // Don't dispatch - keep quantity at 1
      }
    }
    
    dispatch({ type: 'UPDATE_QUANTITY', id, quantity });
  };

  const clearCart = () => {
    dispatch({ type: 'CLEAR_CART' });
  };

  const hasItem = (itemId: string, itemType: string) => {
    return state.items.some(item => item.itemId === itemId && item.itemType === itemType);
  };

  const itemCount = state.items.reduce((total, item) => total + item.quantity, 0);
  const totalCents = state.items.reduce((total, item) => total + (item.itemPrice * item.quantity), 0);

  return (
    <CartContext.Provider value={{
      items: state.items,
      itemCount,
      totalCents,
      sessionId: state.sessionId,
      isLoading: state.isLoading,
      addItem,
      removeItem,
      updateQuantity,
      clearCart,
      hasItem,
      canAddItem
    }}>
      {children}
    </CartContext.Provider>
  );
}

export function useCart() {
  const context = useContext(CartContext);
  if (!context) {
    throw new Error('useCart must be used within a CartProvider');
  }
  return context;
}