import React, { createContext, useContext, useEffect, useState } from 'react';
import { 
  User, 
  signInWithEmailAndPassword, 
  createUserWithEmailAndPassword,
  signOut,
  onAuthStateChanged,
  updateProfile
} from 'firebase/auth';
import { auth } from '@/lib/firebase';
import { toast } from 'sonner';

interface AuthContextType {
  currentUser: User | null;
  login: (email: string, password: string) => Promise<void>;
  signup: (email: string, password: string, displayName: string) => Promise<void>;
  logout: () => Promise<void>;
  loading: boolean;
  isPaidUser: boolean;
  isOwnerUser: boolean;
}

const AuthContext = createContext<AuthContextType>({} as AuthContextType);

export function useAuth() {
  return useContext(AuthContext);
}

export function AuthProvider({ children }: { children: React.ReactNode }) {
  const [currentUser, setCurrentUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [isPaidUser, setIsPaidUser] = useState(false);
  const [isOwnerUser, setIsOwnerUser] = useState(false);
  const [serverDataLoaded, setServerDataLoaded] = useState(false);

  function signup(email: string, password: string, displayName: string) {
    return createUserWithEmailAndPassword(auth, email, password)
      .then((userCredential) => {
        return updateProfile(userCredential.user, { displayName });
      });
  }

  async function login(email: string, password: string) {
    await signInWithEmailAndPassword(auth, email, password);
  }

  function logout() {
    return signOut(auth);
  }

  // Fetch user data from server when user changes
  useEffect(() => {
    const fetchUserData = async () => {
      if (!currentUser) {
        setIsOwnerUser(false);
        setIsPaidUser(false);
        setServerDataLoaded(true);
        return;
      }

      // Set timeout to prevent hanging
      const timeoutId = setTimeout(() => {
        console.warn('Server data fetch timeout - proceeding with defaults');
        setIsOwnerUser(false);
        setIsPaidUser(false);
        setServerDataLoaded(true);
      }, 5000); // 5 second timeout

      try {
        const token = await currentUser.getIdToken();
        const response = await fetch('/api/auth/me', {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          }
        });

        clearTimeout(timeoutId); // Clear timeout if request completes

        if (response.ok) {
          const userData = await response.json();
          setIsPaidUser(userData.isPaid || false);
          setIsOwnerUser(userData.isOwner || false);
          
          // Log for debugging (remove in production)
          if (import.meta.env.MODE === 'development') {
            console.log(`🔐 Server Auth Status - Email: ${userData.email}, Owner: ${userData.isOwner}, Paid: ${userData.isPaid}`);
          }
        } else {
          console.warn('Failed to fetch user data from server:', response.status);
          // Fallback to false for security
          setIsOwnerUser(false);
          setIsPaidUser(false);
        }
      } catch (error) {
        clearTimeout(timeoutId); // Clear timeout if request fails
        console.error('Error fetching user data:', error);
        // Fallback to false for security
        setIsOwnerUser(false);
        setIsPaidUser(false);
      } finally {
        setServerDataLoaded(true);
      }
    };

    fetchUserData();
  }, [currentUser]);

  useEffect(() => {
    const unsubscribe = onAuthStateChanged(auth, (user) => {
      setCurrentUser(user);
      if (!user) {
        // User logged out, reset server data state
        setServerDataLoaded(false);
      }
      setLoading(false);
    });

    return unsubscribe;
  }, []);

  // Show loading spinner only while Firebase auth is initializing
  // Allow app to render even if server data is still loading or failed
  const value = {
    currentUser,
    login,
    signup,
    logout,
    loading,
    isPaidUser,
    isOwnerUser
  };

  // Only show loading during Firebase auth initialization
  // Always render children once Firebase auth state is determined
  if (loading) {
    return (
      <AuthContext.Provider value={value}>
        <div className="min-h-screen bg-background flex items-center justify-center">
          <div className="text-center space-y-4">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary mx-auto"></div>
            <p className="text-muted-foreground">Loading...</p>
          </div>
        </div>
      </AuthContext.Provider>
    );
  }

  return (
    <AuthContext.Provider value={value}>
      {children}
    </AuthContext.Provider>
  );
}
