import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { SwotBoard } from "./SwotBoard";
import { PersonaForm } from "./PersonaForm";
import { MarketSizeFormV2 } from "./MarketSizeFormV2";
import { PricingTableFormV2 } from "./PricingTableFormV2";
import { Layout, Target, User, BarChart3, DollarSign, FileText } from "lucide-react";
import { usePlanStoreV2 } from "@/stores/usePlanStore.v2";
import { useToast } from "@/hooks/use-toast";
import type { SectionId } from "@shared/plan-types";
import type { VisualTemplateType } from "@shared/visuals";
import { VISUAL_TEMPLATES } from "@shared/visuals";

interface TemplatesPanelProps {
  selectedSectionId?: SectionId;
  onSectionChange?: (sectionId: SectionId) => void;
}

export function TemplatesPanel({ 
  selectedSectionId, 
  onSectionChange 
}: TemplatesPanelProps) {
  const [activeTemplate, setActiveTemplate] = useState<VisualTemplateType>('swot');
  const { sections, updateSection } = usePlanStoreV2();
  const { toast } = useToast();

  const availableSections = sections.filter(section => 
    section.kind !== 'financials' // Exclude financial sections as they have their own interface
  );

  const handleContentInsert = (markdown: string) => {
    if (!selectedSectionId) {
      toast({
        title: "No Section Selected",
        description: "Please select a section to insert the template content into.",
        variant: "destructive",
      });
      return;
    }

    const currentSection = sections.find(s => s.id === selectedSectionId);
    if (!currentSection) {
      toast({
        title: "Section Not Found",
        description: "The selected section could not be found.",
        variant: "destructive",
      });
      return;
    }

    // Append the new content to existing content with proper spacing
    const existingContent = currentSection.content || '';
    const separator = existingContent.trim() ? '\n\n' : '';
    const newContent = existingContent + separator + markdown;

    updateSection(selectedSectionId, { content: newContent });
    
    toast({
      title: "Content Inserted",
      description: `Template content has been added to "${currentSection.title}".`,
    });
  };

  const getTemplateIcon = (templateType: VisualTemplateType) => {
    switch (templateType) {
      case 'swot':
        return <Target className="w-4 h-4" />;
      case 'persona':
        return <User className="w-4 h-4" />;
      case 'tam-sam-som':
        return <BarChart3 className="w-4 h-4" />;
      case 'pricing':
        return <DollarSign className="w-4 h-4" />;
      default:
        return <FileText className="w-4 h-4" />;
    }
  };

  return (
    <div className="h-full flex flex-col" data-testid="templates-panel">
      {/* Header */}
      <Card className="mb-4">
        <CardHeader className="pb-3">
          <CardTitle className="text-lg flex items-center gap-2">
            <Layout className="w-5 h-5 text-orange-500" />
            Visual Templates
          </CardTitle>
          <p className="text-sm text-muted-foreground">
            Create structured content for your business plan with visual templates
          </p>
        </CardHeader>
        <CardContent className="space-y-4">
          {/* Section Selector */}
          <div className="space-y-2">
            <label className="text-sm font-medium">Target Section</label>
            <Select 
              value={selectedSectionId || ''} 
              onValueChange={onSectionChange}
              data-testid="section-selector"
            >
              <SelectTrigger>
                <SelectValue placeholder="Select a section to insert content into..." />
              </SelectTrigger>
              <SelectContent>
                {availableSections.map((section) => (
                  <SelectItem 
                    key={section.id} 
                    value={section.id}
                    data-testid={`section-option-${section.id}`}
                  >
                    <div className="flex items-center gap-2">
                      <span className="text-lg">{getSectionIcon(section.kind)}</span>
                      <span>{section.title}</span>
                      {section.content && (
                        <Badge variant="secondary" className="text-xs ml-auto">
                          {section.content.length} chars
                        </Badge>
                      )}
                    </div>
                  </SelectItem>
                ))}
                {availableSections.length === 0 && (
                  <SelectItem value="no-sections" disabled>
                    No sections available
                  </SelectItem>
                )}
              </SelectContent>
            </Select>
            {selectedSectionId && (
              <p className="text-xs text-muted-foreground">
                Template content will be inserted into the selected section
              </p>
            )}
          </div>

          {/* Template Selector */}
          <div className="space-y-2">
            <label className="text-sm font-medium">Visual Template</label>
            <Tabs 
              value={activeTemplate} 
              onValueChange={(value) => setActiveTemplate(value as VisualTemplateType)}
              className="w-full"
              data-testid="template-tabs"
            >
              <TabsList className="grid w-full grid-cols-2 gap-1">
                {VISUAL_TEMPLATES.map((template) => (
                  <TabsTrigger 
                    key={template.key} 
                    value={template.key}
                    className="flex items-center gap-1 text-xs"
                    data-testid={`template-tab-${template.key}`}
                  >
                    {getTemplateIcon(template.key)}
                    <span className="hidden sm:inline">{template.name}</span>
                    <span className="sm:hidden">{template.icon}</span>
                  </TabsTrigger>
                ))}
              </TabsList>
            </Tabs>
          </div>
        </CardContent>
      </Card>

      {/* Template Content */}
      <div className="flex-1 overflow-auto">
        <Tabs value={activeTemplate} onValueChange={(value) => setActiveTemplate(value as VisualTemplateType)}>
          {/* SWOT Analysis */}
          <TabsContent value="swot" className="mt-0">
            <SwotBoard 
              onInsert={handleContentInsert}
              data-testid="swot-template-content"
            />
          </TabsContent>

          {/* Customer Persona */}
          <TabsContent value="persona" className="mt-0">
            <PersonaForm 
              onInsert={handleContentInsert}
              data-testid="persona-template-content"
            />
          </TabsContent>

          {/* TAM/SAM/SOM Market Sizing V2 */}
          <TabsContent value="tam-sam-som" className="mt-0">
            <MarketSizeFormV2 
              onInsert={handleContentInsert}
              targetSectionId={selectedSectionId || ""}
              data-testid="tam-sam-som-template-content"
            />
          </TabsContent>

          {/* Pricing Strategy V2 */}
          <TabsContent value="pricing" className="mt-0">
            <PricingTableFormV2 
              onInsert={handleContentInsert}
              targetSectionId={selectedSectionId || ""}
              data-testid="pricing-template-content"
            />
          </TabsContent>
        </Tabs>
      </div>

      {/* Instructions */}
      <Card className="mt-4 border-dashed">
        <CardContent className="pt-4">
          <div className="text-sm text-muted-foreground space-y-2">
            <p><strong>How to use Visual Templates:</strong></p>
            <ol className="list-decimal list-inside space-y-1 ml-2 text-xs">
              <li>Select a target section from the dropdown above</li>
              <li>Choose a visual template (SWOT, Persona, etc.)</li>
              <li>Fill in the template fields with your business information</li>
              <li>Click "Insert into Section" to add the formatted content</li>
              <li>Switch to the Editor tab to review and edit the inserted content</li>
            </ol>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}

// Helper function to get section icons (matching the main builder)
function getSectionIcon(kind?: string) {
  switch (kind) {
    case 'executive-summary':
      return '📊';
    case 'company-overview':
      return '🏢';
    case 'products-services':
      return '💼';
    case 'market-analysis':
      return '📈';
    case 'marketing-sales':
      return '📢';
    case 'operations':
      return '⚙️';
    case 'org-management':
      return '👥';
    case 'financials':
      return '💰';
    default:
      return '📄';
  }
}