import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { BarChart3, Target, Zap, FileText } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import type { TamSamSomData } from "@shared/visuals";
import { DEFAULT_TAM_SAM_SOM_DATA } from "@shared/visuals";
import { renderTamSamSomMarkdown, validateTamSamSomData } from "@/lib/visualsRender";

interface TamSamSomFormProps {
  data?: TamSamSomData;
  onInsert?: (markdown: string) => void;
  onDataChange?: (data: TamSamSomData) => void;
}

export function TamSamSomForm({ 
  data = DEFAULT_TAM_SAM_SOM_DATA, 
  onInsert, 
  onDataChange 
}: TamSamSomFormProps) {
  const [marketData, setMarketData] = useState<TamSamSomData>(data);
  const [validationErrors, setValidationErrors] = useState<string[]>([]);
  const { toast } = useToast();

  const updateMarketData = (field: keyof TamSamSomData, value: string) => {
    const updatedData = { ...marketData, [field]: value };
    setMarketData(updatedData);
    onDataChange?.(updatedData);
    
    if (validationErrors.length > 0) {
      setValidationErrors([]);
    }
  };

  const handleInsert = () => {
    const errors = validateTamSamSomData(marketData);
    if (errors.length > 0) {
      setValidationErrors(errors);
      toast({
        title: "Validation Error",
        description: errors.join(", "),
        variant: "destructive",
      });
      return;
    }

    const fieldsWithContent = Object.values(marketData).filter(value => value.trim().length > 0);
    if (fieldsWithContent.length === 0) {
      toast({
        title: "Empty Market Sizing",
        description: "Please fill in at least one market size before inserting.",
        variant: "destructive",
      });
      return;
    }

    const markdown = renderTamSamSomMarkdown(marketData);
    onInsert?.(markdown);
    toast({
      title: "Market Sizing Inserted",
      description: "The TAM/SAM/SOM analysis has been added to your section.",
    });
  };

  const getCompletionPercentage = () => {
    const totalFields = Object.keys(marketData).length;
    const completedFields = Object.values(marketData).filter(value => value.trim().length > 0).length;
    return Math.round((completedFields / totalFields) * 100);
  };

  const completionPercentage = getCompletionPercentage();

  return (
    <div className="space-y-6" data-testid="tam-sam-som-form">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <BarChart3 className="w-5 h-5 text-orange-500" />
            TAM/SAM/SOM Market Sizing
          </h3>
          <p className="text-sm text-muted-foreground">
            Define your total addressable, serviceable, and obtainable markets
          </p>
        </div>
        <div className="flex items-center gap-2">
          <Badge variant="outline" data-testid="tam-completion-percentage">
            {completionPercentage}% complete
          </Badge>
          <Button 
            onClick={handleInsert}
            disabled={completionPercentage === 0}
            className="bg-orange-500 hover:bg-orange-600"
            data-testid="tam-insert-button"
          >
            Insert into Section
          </Button>
        </div>
      </div>

      {/* Validation Errors */}
      {validationErrors.length > 0 && (
        <Card className="border-destructive bg-destructive/5">
          <CardContent className="pt-4">
            <div className="text-sm text-destructive space-y-1">
              <p><strong>Please fix the following errors:</strong></p>
              <ul className="list-disc list-inside ml-2">
                {validationErrors.map((error, index) => (
                  <li key={index}>{error}</li>
                ))}
              </ul>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Form Fields */}
      <div className="grid gap-6">
        {/* TAM */}
        <Card data-testid="tam-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <Target className="w-4 h-4 text-blue-500" />
              TAM (Total Addressable Market)
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="tam-input">The entire demand for your product or service</Label>
              <Textarea
                id="tam-input"
                placeholder="e.g., 'The global project management software market is valued at $6.68 billion annually, with an expected CAGR of 10.67% through 2028.'"
                value={marketData.tam}
                onChange={(e) => updateMarketData('tam', e.target.value)}
                rows={3}
                data-testid="tam-textarea"
              />
              <p className="text-xs text-muted-foreground">
                Include market size, growth rates, and geographic scope
              </p>
            </div>
          </CardContent>
        </Card>

        {/* SAM */}
        <Card data-testid="sam-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <Zap className="w-4 h-4 text-green-500" />
              SAM (Serviceable Available Market)
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="sam-input">The portion of TAM you can actually reach</Label>
              <Textarea
                id="sam-input"
                placeholder="e.g., 'Small to medium businesses (10-500 employees) in North America using cloud-based tools, representing $2.1 billion of the total market.'"
                value={marketData.sam}
                onChange={(e) => updateMarketData('sam', e.target.value)}
                rows={3}
                data-testid="sam-textarea"
              />
              <p className="text-xs text-muted-foreground">
                Factor in geographic limitations, product fit, and distribution channels
              </p>
            </div>
          </CardContent>
        </Card>

        {/* SOM */}
        <Card data-testid="som-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <Target className="w-4 h-4 text-purple-500" />
              SOM (Serviceable Obtainable Market)
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="som-input">The market share you can realistically capture</Label>
              <Textarea
                id="som-input"
                placeholder="e.g., 'Targeting 0.5% market share within 3 years, representing $10.5M in annual revenue based on competitive positioning and go-to-market strategy.'"
                value={marketData.som}
                onChange={(e) => updateMarketData('som', e.target.value)}
                rows={3}
                data-testid="som-textarea"
              />
              <p className="text-xs text-muted-foreground">
                Consider competitive landscape, resources, and realistic growth projections
              </p>
            </div>
          </CardContent>
        </Card>

        {/* Sources */}
        <Card data-testid="sources-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <FileText className="w-4 h-4 text-orange-500" />
              Sources & Assumptions
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="sources-input">Data sources and key assumptions</Label>
              <Textarea
                id="sources-input"
                placeholder="e.g., 'Data from Gartner Research 2024, Grand View Research, internal customer surveys. Assumes 15% annual growth in SMB cloud adoption.'"
                value={marketData.sources}
                onChange={(e) => updateMarketData('sources', e.target.value)}
                rows={3}
                data-testid="sources-textarea"
              />
              <p className="text-xs text-muted-foreground">
                Include research sources, surveys, and key assumptions made in calculations
              </p>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Instructions */}
      <Card className="border-dashed">
        <CardContent className="pt-4">
          <div className="text-sm text-muted-foreground space-y-2">
            <p><strong>Market sizing framework:</strong></p>
            <ul className="list-disc list-inside space-y-1 ml-2">
              <li><strong>TAM:</strong> Start with the broadest possible market size</li>
              <li><strong>SAM:</strong> Narrow down based on your product's actual reach</li>
              <li><strong>SOM:</strong> Be realistic about what you can achieve with your resources</li>
              <li>Use credible sources like industry reports, government data, or primary research</li>
              <li>Update your market sizing regularly as you gather more data</li>
            </ul>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}