import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Plus, X, Target, AlertTriangle, TrendingUp, Shield } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import type { SwotData } from "@shared/visuals";
import { DEFAULT_SWOT_DATA } from "@shared/visuals";
import { renderSwotMarkdown, validateSwotData } from "@/lib/visualsRender";

interface SwotBoardProps {
  data?: SwotData;
  onInsert?: (markdown: string) => void;
  onDataChange?: (data: SwotData) => void;
}

export function SwotBoard({ 
  data = DEFAULT_SWOT_DATA, 
  onInsert, 
  onDataChange 
}: SwotBoardProps) {
  const [swotData, setSwotData] = useState<SwotData>(data);
  const [newItems, setNewItems] = useState({
    strengths: "",
    weaknesses: "",
    opportunities: "",
    threats: ""
  });
  const { toast } = useToast();

  const updateSwotData = (updatedData: SwotData) => {
    setSwotData(updatedData);
    onDataChange?.(updatedData);
  };

  const addItem = (category: keyof SwotData, item: string) => {
    if (!item.trim()) return;
    
    const updatedData = {
      ...swotData,
      [category]: [...swotData[category], item.trim()]
    };
    updateSwotData(updatedData);
    setNewItems(prev => ({ ...prev, [category]: "" }));
  };

  const removeItem = (category: keyof SwotData, index: number) => {
    const updatedData = {
      ...swotData,
      [category]: swotData[category].filter((_, i) => i !== index)
    };
    updateSwotData(updatedData);
  };

  const handleInsert = () => {
    const errors = validateSwotData(swotData);
    if (errors.length > 0) {
      toast({
        title: "Validation Error",
        description: errors.join(", "),
        variant: "destructive",
      });
      return;
    }

    const markdown = renderSwotMarkdown(swotData);
    onInsert?.(markdown);
    toast({
      title: "SWOT Analysis Inserted",
      description: "The SWOT analysis has been added to your section.",
    });
  };

  const totalItems = Object.values(swotData).flat().length;

  const QuadrantCard = ({ 
    title, 
    category, 
    items, 
    icon: Icon, 
    colorClass,
    bgClass 
  }: {
    title: string;
    category: keyof SwotData;
    items: string[];
    icon: any;
    colorClass: string;
    bgClass: string;
  }) => (
    <Card className={`h-full ${bgClass}`} data-testid={`swot-quadrant-${category}`}>
      <CardHeader className="pb-3">
        <CardTitle className={`text-lg flex items-center gap-2 ${colorClass}`}>
          <Icon className="w-5 h-5" />
          {title}
          <Badge variant="secondary" className="ml-auto">
            {items.length}
          </Badge>
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-3">
        {/* Existing Items */}
        <div className="space-y-2 max-h-40 overflow-y-auto" data-testid={`${category}-items`}>
          {items.map((item, index) => (
            <div 
              key={index}
              className="flex items-start gap-2 p-2 bg-background/50 rounded border text-sm"
              data-testid={`${category}-item-${index}`}
            >
              <span className="flex-1">{item}</span>
              <Button
                size="sm"
                variant="ghost"
                onClick={() => removeItem(category, index)}
                className="h-6 w-6 p-0 hover:bg-destructive hover:text-destructive-foreground"
                data-testid={`${category}-remove-${index}`}
              >
                <X className="w-3 h-3" />
              </Button>
            </div>
          ))}
          {items.length === 0 && (
            <p className="text-muted-foreground text-sm italic">
              No {title.toLowerCase()} added yet
            </p>
          )}
        </div>

        {/* Add New Item */}
        <div className="flex gap-2">
          <Input
            placeholder={`Add ${title.toLowerCase().slice(0, -1)}...`}
            value={newItems[category]}
            onChange={(e) => setNewItems(prev => ({ ...prev, [category]: e.target.value }))}
            onKeyDown={(e) => {
              if (e.key === 'Enter') {
                addItem(category, newItems[category]);
              }
            }}
            className="text-sm"
            data-testid={`${category}-input`}
          />
          <Button
            size="sm"
            onClick={() => addItem(category, newItems[category])}
            disabled={!newItems[category].trim()}
            data-testid={`${category}-add-button`}
          >
            <Plus className="w-4 h-4" />
          </Button>
        </div>
      </CardContent>
    </Card>
  );

  return (
    <div className="space-y-6" data-testid="swot-board">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Target className="w-5 h-5 text-orange-500" />
            SWOT Analysis Builder
          </h3>
          <p className="text-sm text-muted-foreground">
            Analyze internal strengths & weaknesses, external opportunities & threats
          </p>
        </div>
        <div className="flex items-center gap-2">
          <Badge variant="outline" data-testid="total-items-count">
            {totalItems} items
          </Badge>
          <Button 
            onClick={handleInsert}
            disabled={totalItems === 0}
            className="bg-orange-500 hover:bg-orange-600"
            data-testid="swot-insert-button"
          >
            Insert into Section
          </Button>
        </div>
      </div>

      {/* 4-Quadrant Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4 h-[600px]">
        {/* Strengths - Top Left */}
        <QuadrantCard
          title="Strengths"
          category="strengths"
          items={swotData.strengths}
          icon={Shield}
          colorClass="text-green-600"
          bgClass="border-green-200 bg-green-50/30"
        />

        {/* Weaknesses - Top Right */}
        <QuadrantCard
          title="Weaknesses"
          category="weaknesses"
          items={swotData.weaknesses}
          icon={AlertTriangle}
          colorClass="text-red-600"
          bgClass="border-red-200 bg-red-50/30"
        />

        {/* Opportunities - Bottom Left */}
        <QuadrantCard
          title="Opportunities"
          category="opportunities"
          items={swotData.opportunities}
          icon={TrendingUp}
          colorClass="text-blue-600"
          bgClass="border-blue-200 bg-blue-50/30"
        />

        {/* Threats - Bottom Right */}
        <QuadrantCard
          title="Threats"
          category="threats"
          items={swotData.threats}
          icon={X}
          colorClass="text-orange-600"
          bgClass="border-orange-200 bg-orange-50/30"
        />
      </div>

      {/* Instructions */}
      <Card className="border-dashed">
        <CardContent className="pt-4">
          <div className="text-sm text-muted-foreground space-y-1">
            <p><strong>Instructions:</strong></p>
            <ul className="list-disc list-inside space-y-1 ml-2">
              <li><strong>Strengths:</strong> Internal positive factors that give you an advantage</li>
              <li><strong>Weaknesses:</strong> Internal negative factors that put you at a disadvantage</li>
              <li><strong>Opportunities:</strong> External positive factors you can capitalize on</li>
              <li><strong>Threats:</strong> External negative factors that could harm your business</li>
            </ul>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}