import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Label } from "@/components/ui/label";
import { Plus, X, DollarSign, Package } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import type { PricingTierData } from "@shared/visuals";
import { DEFAULT_PRICING_TIERS } from "@shared/visuals";
import { renderPricingMarkdown, validatePricingData } from "@/lib/visualsRender";

interface PricingFormProps {
  data?: PricingTierData[];
  onInsert?: (markdown: string) => void;
  onDataChange?: (data: PricingTierData[]) => void;
}

export function PricingForm({ 
  data = DEFAULT_PRICING_TIERS, 
  onInsert, 
  onDataChange 
}: PricingFormProps) {
  const [pricingTiers, setPricingTiers] = useState<PricingTierData[]>(data);
  const [validationErrors, setValidationErrors] = useState<string[]>([]);
  const { toast } = useToast();

  const updatePricingTiers = (updatedTiers: PricingTierData[]) => {
    setPricingTiers(updatedTiers);
    onDataChange?.(updatedTiers);
    
    if (validationErrors.length > 0) {
      setValidationErrors([]);
    }
  };

  const updateTier = (index: number, field: keyof PricingTierData, value: string | string[]) => {
    const updatedTiers = pricingTiers.map((tier, i) => 
      i === index ? { ...tier, [field]: value } : tier
    );
    updatePricingTiers(updatedTiers);
  };

  const addTier = () => {
    const newTier: PricingTierData = {
      name: `Tier ${pricingTiers.length + 1}`,
      price: "$0/month",
      features: []
    };
    updatePricingTiers([...pricingTiers, newTier]);
  };

  const removeTier = (index: number) => {
    if (pricingTiers.length <= 1) {
      toast({
        title: "Cannot Remove Tier",
        description: "You need at least one pricing tier.",
        variant: "destructive",
      });
      return;
    }
    updatePricingTiers(pricingTiers.filter((_, i) => i !== index));
  };

  const addFeature = (tierIndex: number, feature: string) => {
    if (!feature.trim()) return;
    
    const updatedTiers = pricingTiers.map((tier, i) => 
      i === tierIndex 
        ? { ...tier, features: [...tier.features, feature.trim()] }
        : tier
    );
    updatePricingTiers(updatedTiers);
  };

  const removeFeature = (tierIndex: number, featureIndex: number) => {
    const updatedTiers = pricingTiers.map((tier, i) => 
      i === tierIndex 
        ? { ...tier, features: tier.features.filter((_, fi) => fi !== featureIndex) }
        : tier
    );
    updatePricingTiers(updatedTiers);
  };

  const handleInsert = () => {
    const errors = validatePricingData(pricingTiers);
    if (errors.length > 0) {
      setValidationErrors(errors);
      toast({
        title: "Validation Error",
        description: errors.join(", "),
        variant: "destructive",
      });
      return;
    }

    const markdown = renderPricingMarkdown(pricingTiers);
    onInsert?.(markdown);
    toast({
      title: "Pricing Table Inserted",
      description: "The pricing strategy has been added to your section.",
    });
  };

  const totalFeatures = pricingTiers.reduce((sum, tier) => sum + tier.features.length, 0);

  return (
    <div className="space-y-6" data-testid="pricing-form">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <DollarSign className="w-5 h-5 text-orange-500" />
            Pricing Strategy Builder
          </h3>
          <p className="text-sm text-muted-foreground">
            Define your pricing tiers and feature offerings
          </p>
        </div>
        <div className="flex items-center gap-2">
          <Badge variant="outline" data-testid="pricing-tiers-count">
            {pricingTiers.length} tiers
          </Badge>
          <Badge variant="outline" data-testid="pricing-features-count">
            {totalFeatures} features
          </Badge>
          <Button 
            onClick={handleInsert}
            disabled={pricingTiers.length === 0}
            className="bg-orange-500 hover:bg-orange-600"
            data-testid="pricing-insert-button"
          >
            Insert into Section
          </Button>
        </div>
      </div>

      {/* Validation Errors */}
      {validationErrors.length > 0 && (
        <Card className="border-destructive bg-destructive/5">
          <CardContent className="pt-4">
            <div className="text-sm text-destructive space-y-1">
              <p><strong>Please fix the following errors:</strong></p>
              <ul className="list-disc list-inside ml-2">
                {validationErrors.map((error, index) => (
                  <li key={index}>{error}</li>
                ))}
              </ul>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Pricing Tiers */}
      <div className="grid gap-6">
        {pricingTiers.map((tier, tierIndex) => (
          <PricingTierCard
            key={tierIndex}
            tier={tier}
            tierIndex={tierIndex}
            onUpdateTier={(field, value) => updateTier(tierIndex, field, value)}
            onRemoveTier={() => removeTier(tierIndex)}
            onAddFeature={(feature) => addFeature(tierIndex, feature)}
            onRemoveFeature={(featureIndex) => removeFeature(tierIndex, featureIndex)}
            canRemove={pricingTiers.length > 1}
          />
        ))}
      </div>

      {/* Add Tier Button */}
      <div className="flex justify-center">
        <Button
          onClick={addTier}
          variant="outline"
          className="flex items-center gap-2"
          data-testid="add-tier-button"
        >
          <Plus className="w-4 h-4" />
          Add Pricing Tier
        </Button>
      </div>

      {/* Instructions */}
      <Card className="border-dashed">
        <CardContent className="pt-4">
          <div className="text-sm text-muted-foreground space-y-2">
            <p><strong>Pricing strategy best practices:</strong></p>
            <ul className="list-disc list-inside space-y-1 ml-2">
              <li>Offer 3-4 tiers maximum to avoid choice paralysis</li>
              <li>Make the middle tier most attractive (anchor pricing)</li>
              <li>Include clear value propositions for each tier</li>
              <li>Consider freemium, trial, or money-back guarantees</li>
              <li>Test pricing with real customers before launch</li>
            </ul>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}

interface PricingTierCardProps {
  tier: PricingTierData;
  tierIndex: number;
  onUpdateTier: (field: keyof PricingTierData, value: string | string[]) => void;
  onRemoveTier: () => void;
  onAddFeature: (feature: string) => void;
  onRemoveFeature: (featureIndex: number) => void;
  canRemove: boolean;
}

function PricingTierCard({
  tier,
  tierIndex,
  onUpdateTier,
  onRemoveTier,
  onAddFeature,
  onRemoveFeature,
  canRemove
}: PricingTierCardProps) {
  const [newFeature, setNewFeature] = useState("");

  const handleAddFeature = () => {
    onAddFeature(newFeature);
    setNewFeature("");
  };

  return (
    <Card data-testid={`pricing-tier-${tierIndex}`}>
      <CardHeader className="pb-3">
        <div className="flex items-center justify-between">
          <CardTitle className="text-base flex items-center gap-2">
            <Package className="w-4 h-4 text-blue-500" />
            Pricing Tier {tierIndex + 1}
          </CardTitle>
          {canRemove && (
            <Button
              size="sm"
              variant="ghost"
              onClick={onRemoveTier}
              className="h-8 w-8 p-0 hover:bg-destructive hover:text-destructive-foreground"
              data-testid={`remove-tier-${tierIndex}`}
            >
              <X className="w-4 h-4" />
            </Button>
          )}
        </div>
      </CardHeader>
      <CardContent className="space-y-4">
        {/* Tier Name and Price */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div className="space-y-2">
            <Label htmlFor={`tier-name-${tierIndex}`}>Tier Name</Label>
            <Input
              id={`tier-name-${tierIndex}`}
              placeholder="e.g., Basic, Pro, Enterprise"
              value={tier.name}
              onChange={(e) => onUpdateTier('name', e.target.value)}
              data-testid={`tier-name-input-${tierIndex}`}
            />
          </div>
          <div className="space-y-2">
            <Label htmlFor={`tier-price-${tierIndex}`}>Price</Label>
            <Input
              id={`tier-price-${tierIndex}`}
              placeholder="e.g., $29/month, Free, Contact us"
              value={tier.price}
              onChange={(e) => onUpdateTier('price', e.target.value)}
              data-testid={`tier-price-input-${tierIndex}`}
            />
          </div>
        </div>

        {/* Features */}
        <div className="space-y-3">
          <Label>Features</Label>
          
          {/* Existing Features */}
          <div className="space-y-2 max-h-40 overflow-y-auto" data-testid={`tier-features-${tierIndex}`}>
            {tier.features.map((feature, featureIndex) => (
              <div 
                key={featureIndex}
                className="flex items-start gap-2 p-2 bg-muted/50 rounded border text-sm"
                data-testid={`tier-${tierIndex}-feature-${featureIndex}`}
              >
                <span className="flex-1">{feature}</span>
                <Button
                  size="sm"
                  variant="ghost"
                  onClick={() => onRemoveFeature(featureIndex)}
                  className="h-6 w-6 p-0 hover:bg-destructive hover:text-destructive-foreground"
                  data-testid={`remove-feature-${tierIndex}-${featureIndex}`}
                >
                  <X className="w-3 h-3" />
                </Button>
              </div>
            ))}
            {tier.features.length === 0 && (
              <p className="text-muted-foreground text-sm italic">
                No features added yet
              </p>
            )}
          </div>

          {/* Add New Feature */}
          <div className="flex gap-2">
            <Input
              placeholder="Add feature..."
              value={newFeature}
              onChange={(e) => setNewFeature(e.target.value)}
              onKeyDown={(e) => {
                if (e.key === 'Enter') {
                  handleAddFeature();
                }
              }}
              className="text-sm"
              data-testid={`add-feature-input-${tierIndex}`}
            />
            <Button
              size="sm"
              onClick={handleAddFeature}
              disabled={!newFeature.trim()}
              data-testid={`add-feature-button-${tierIndex}`}
            >
              <Plus className="w-4 h-4" />
            </Button>
          </div>
        </div>
      </CardContent>
    </Card>
  );
}