import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { User, Users, Target, Frown, TrendingUp } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import type { PersonaData } from "@shared/visuals";
import { DEFAULT_PERSONA_DATA } from "@shared/visuals";
import { renderPersonaMarkdown, validatePersonaData } from "@/lib/visualsRender";

interface PersonaFormProps {
  data?: PersonaData;
  onInsert?: (markdown: string) => void;
  onDataChange?: (data: PersonaData) => void;
}

export function PersonaForm({ 
  data = DEFAULT_PERSONA_DATA, 
  onInsert, 
  onDataChange 
}: PersonaFormProps) {
  const [personaData, setPersonaData] = useState<PersonaData>(data);
  const [validationErrors, setValidationErrors] = useState<string[]>([]);
  const { toast } = useToast();

  const updatePersonaData = (field: keyof PersonaData, value: string) => {
    const updatedData = { ...personaData, [field]: value };
    setPersonaData(updatedData);
    onDataChange?.(updatedData);
    
    // Clear validation errors when user types
    if (validationErrors.length > 0) {
      setValidationErrors([]);
    }
  };

  const handleInsert = () => {
    const errors = validatePersonaData(personaData);
    if (errors.length > 0) {
      setValidationErrors(errors);
      toast({
        title: "Validation Error",
        description: errors.join(", "),
        variant: "destructive",
      });
      return;
    }

    // Check if at least some fields are filled
    const fieldsWithContent = Object.values(personaData).filter(value => value.trim().length > 0);
    if (fieldsWithContent.length === 0) {
      toast({
        title: "Empty Persona",
        description: "Please fill in at least one field before inserting.",
        variant: "destructive",
      });
      return;
    }

    const markdown = renderPersonaMarkdown(personaData);
    onInsert?.(markdown);
    toast({
      title: "Customer Persona Inserted",
      description: "The customer persona has been added to your section.",
    });
  };

  const getCompletionPercentage = () => {
    const totalFields = Object.keys(personaData).length;
    const completedFields = Object.values(personaData).filter(value => value.trim().length > 0).length;
    return Math.round((completedFields / totalFields) * 100);
  };

  const completionPercentage = getCompletionPercentage();

  return (
    <div className="space-y-6" data-testid="persona-form">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <User className="w-5 h-5 text-orange-500" />
            Customer Persona Builder
          </h3>
          <p className="text-sm text-muted-foreground">
            Define your ideal customer profile with detailed characteristics
          </p>
        </div>
        <div className="flex items-center gap-2">
          <Badge variant="outline" data-testid="completion-percentage">
            {completionPercentage}% complete
          </Badge>
          <Button 
            onClick={handleInsert}
            disabled={completionPercentage === 0}
            className="bg-orange-500 hover:bg-orange-600"
            data-testid="persona-insert-button"
          >
            Insert into Section
          </Button>
        </div>
      </div>

      {/* Validation Errors */}
      {validationErrors.length > 0 && (
        <Card className="border-destructive bg-destructive/5">
          <CardContent className="pt-4">
            <div className="text-sm text-destructive space-y-1">
              <p><strong>Please fix the following errors:</strong></p>
              <ul className="list-disc list-inside ml-2">
                {validationErrors.map((error, index) => (
                  <li key={index}>{error}</li>
                ))}
              </ul>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Form Fields */}
      <div className="grid gap-6">
        {/* Persona Name */}
        <Card data-testid="persona-name-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <Users className="w-4 h-4 text-blue-500" />
              Persona Name
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="persona-name">Give your persona a name</Label>
              <Input
                id="persona-name"
                placeholder="e.g., 'Marketing Manager Mary' or 'Tech-Savvy Tom'"
                value={personaData.name}
                onChange={(e) => updatePersonaData('name', e.target.value)}
                data-testid="persona-name-input"
              />
              <p className="text-xs text-muted-foreground">
                A memorable name helps humanize your target customer
              </p>
            </div>
          </CardContent>
        </Card>

        {/* Demographics */}
        <Card data-testid="persona-demographics-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <Users className="w-4 h-4 text-green-500" />
              Demographics
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="persona-demographics">Age, gender, location, income, education</Label>
              <Textarea
                id="persona-demographics"
                placeholder="e.g., '25-35 years old, female, urban professionals earning $50-75k annually with college degrees'"
                value={personaData.demographics}
                onChange={(e) => updatePersonaData('demographics', e.target.value)}
                rows={3}
                data-testid="persona-demographics-input"
              />
              <p className="text-xs text-muted-foreground">
                Include key demographic characteristics that define your target market
              </p>
            </div>
          </CardContent>
        </Card>

        {/* Goals & Objectives */}
        <Card data-testid="persona-goals-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <Target className="w-4 h-4 text-purple-500" />
              Goals & Objectives
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="persona-goals">What does this persona want to achieve?</Label>
              <Textarea
                id="persona-goals"
                placeholder="e.g., 'Increase team productivity, reduce manual work, advance in career, save time'"
                value={personaData.goals}
                onChange={(e) => updatePersonaData('goals', e.target.value)}
                rows={3}
                data-testid="persona-goals-input"
              />
              <p className="text-xs text-muted-foreground">
                Focus on both professional and personal objectives that drive their decisions
              </p>
            </div>
          </CardContent>
        </Card>

        {/* Frustrations & Pain Points */}
        <Card data-testid="persona-frustrations-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <Frown className="w-4 h-4 text-red-500" />
              Frustrations & Pain Points
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="persona-frustrations">What problems keep them up at night?</Label>
              <Textarea
                id="persona-frustrations"
                placeholder="e.g., 'Too many repetitive tasks, unclear team communication, budget constraints, lack of good tools'"
                value={personaData.frustrations}
                onChange={(e) => updatePersonaData('frustrations', e.target.value)}
                rows={3}
                data-testid="persona-frustrations-input"
              />
              <p className="text-xs text-muted-foreground">
                Understanding pain points helps you position your solution effectively
              </p>
            </div>
          </CardContent>
        </Card>

        {/* Buying Triggers */}
        <Card data-testid="persona-triggers-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <TrendingUp className="w-4 h-4 text-orange-500" />
              Buying Triggers & Motivators
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="persona-triggers">What motivates them to purchase?</Label>
              <Textarea
                id="persona-triggers"
                placeholder="e.g., 'ROI calculations, peer recommendations, free trials, time pressure, competitor threats'"
                value={personaData.buyingTriggers}
                onChange={(e) => updatePersonaData('buyingTriggers', e.target.value)}
                rows={3}
                data-testid="persona-triggers-input"
              />
              <p className="text-xs text-muted-foreground">
                Identify the key factors that push them from consideration to purchase
              </p>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Instructions */}
      <Card className="border-dashed">
        <CardContent className="pt-4">
          <div className="text-sm text-muted-foreground space-y-2">
            <p><strong>Tips for creating effective personas:</strong></p>
            <ul className="list-disc list-inside space-y-1 ml-2">
              <li>Base personas on real customer research and data when possible</li>
              <li>Focus on behaviors and motivations, not just demographics</li>
              <li>Keep descriptions specific and actionable</li>
              <li>Create 2-3 primary personas rather than many generic ones</li>
              <li>Update personas regularly as you learn more about your customers</li>
            </ul>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}