import { useState, useRef } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { PieChart, Pie, Cell, ResponsiveContainer, Tooltip, Legend } from "recharts";
import { BarChart3, Target, Zap, FileText, PieChart as PieChartIcon } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { useBrandKitStore } from "@/stores/useBrandKitStore";
import { getBrandPalette } from "@/lib/brand";
import type { TamSamSomData } from "@shared/visuals";
import { DEFAULT_TAM_SAM_SOM_DATA } from "@shared/visuals";
import { renderTamSamSomMarkdown, validateTamSamSomData } from "@/lib/visualsRender";
import { ChartSnapshotButtonV2 } from "@/components/ChartSnapshotButtonV2";

interface MarketSizeFormV2Props {
  data?: TamSamSomData;
  onInsert?: (markdown: string) => void;
  onDataChange?: (data: TamSamSomData) => void;
  targetSectionId?: string;
}

// Enhanced market data with numeric values for visualization
interface EnhancedMarketData extends TamSamSomData {
  tamValue: number;
  samValue: number;
  somValue: number;
}

export function MarketSizeFormV2({ 
  data = DEFAULT_TAM_SAM_SOM_DATA, 
  onInsert, 
  onDataChange,
  targetSectionId = ""
}: MarketSizeFormV2Props) {
  const [marketData, setMarketData] = useState<EnhancedMarketData>({
    ...data,
    tamValue: 1000,
    samValue: 300,
    somValue: 50
  });
  const [validationErrors, setValidationErrors] = useState<string[]>([]);
  const { toast } = useToast();
  const brand = useBrandKitStore((s) => s.getCurrentKit());
  const chartRef = useRef<HTMLDivElement>(null);
  
  // Get brand-aware color palette
  const palette = getBrandPalette(brand);
  
  const COLORS = {
    tam: palette[0], // primary color
    sam: palette[1], // secondary color  
    som: palette[2]  // accent color
  };

  const updateMarketData = (field: keyof EnhancedMarketData, value: string | number) => {
    const updatedData = { ...marketData, [field]: value };
    setMarketData(updatedData);
    onDataChange?.(updatedData);
    
    if (validationErrors.length > 0) {
      setValidationErrors([]);
    }
  };

  const handleInsert = () => {
    const errors = validateTamSamSomData(marketData);
    if (errors.length > 0) {
      setValidationErrors(errors);
      toast({
        title: "Validation Error",
        description: errors.join(", "),
        variant: "destructive",
      });
      return;
    }

    const fieldsWithContent = Object.values({
      tam: marketData.tam,
      sam: marketData.sam,
      som: marketData.som,
      sources: marketData.sources
    }).filter(value => value.trim().length > 0);
    
    if (fieldsWithContent.length === 0) {
      toast({
        title: "Empty Market Sizing",
        description: "Please fill in at least one market size before inserting.",
        variant: "destructive",
      });
      return;
    }

    const markdown = renderTamSamSomMarkdown(marketData);
    onInsert?.(markdown);
    toast({
      title: "Market Sizing Inserted",
      description: "The TAM/SAM/SOM analysis has been added to your section.",
    });
  };

  const getCompletionPercentage = () => {
    const textFields = [marketData.tam, marketData.sam, marketData.som, marketData.sources];
    const completedFields = textFields.filter(value => value.trim().length > 0).length;
    return Math.round((completedFields / textFields.length) * 100);
  };

  const chartData = [
    { name: "TAM", value: marketData.tamValue, description: "Total Addressable Market" },
    { name: "SAM", value: marketData.samValue, description: "Serviceable Available Market" },
    { name: "SOM", value: marketData.somValue, description: "Serviceable Obtainable Market" }
  ];

  const completionPercentage = getCompletionPercentage();

  return (
    <div className="space-y-6" data-testid="market-size-form-v2">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <BarChart3 className="w-5 h-5 text-orange-500" />
            TAM/SAM/SOM Market Sizing V2
          </h3>
          <p className="text-sm text-muted-foreground">
            Define your market sizes with visual chart preview
          </p>
        </div>
        <div className="flex items-center gap-2">
          <Badge variant="outline" data-testid="market-completion-percentage">
            {completionPercentage}% complete
          </Badge>
          <Button 
            onClick={handleInsert}
            disabled={completionPercentage === 0}
            className="bg-orange-500 hover:bg-orange-600"
            data-testid="market-insert-button"
          >
            Insert as Text
          </Button>
        </div>
      </div>

      {/* Validation Errors */}
      {validationErrors.length > 0 && (
        <Card className="border-destructive bg-destructive/5">
          <CardContent className="pt-4">
            <div className="text-sm text-destructive space-y-1">
              <p><strong>Please fix the following errors:</strong></p>
              <ul className="list-disc list-inside ml-2">
                {validationErrors.map((error, index) => (
                  <li key={index}>{error}</li>
                ))}
              </ul>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Chart Preview */}
      <Card data-testid="market-chart-preview">
        <CardHeader className="pb-3">
          <div className="flex items-center justify-between">
            <CardTitle className="text-base flex items-center gap-2">
              <PieChartIcon className="w-4 h-4 text-blue-500" />
              Market Size Visualization
            </CardTitle>
            {targetSectionId && (
              <ChartSnapshotButtonV2
                targetSectionId={targetSectionId}
                getElement={() => chartRef.current}
                title="Market Size (TAM/SAM/SOM)"
                className="scale-90"
              />
            )}
          </div>
        </CardHeader>
        <CardContent>
          <div ref={chartRef} className="bg-white rounded-lg p-4" style={{ fontFamily: brand?.fontBody || "Inter, ui-sans-serif" }}>
            <ResponsiveContainer width="100%" height={300}>
              <PieChart>
                <Pie
                  data={chartData}
                  cx="50%"
                  cy="50%"
                  outerRadius={100}
                  fill="#8884d8"
                  dataKey="value"
                  label={({ name, value }) => `${name}: ${value}`}
                >
                  {chartData.map((entry, index) => (
                    <Cell key={`cell-${index}`} fill={palette[index % palette.length]} />
                  ))}
                </Pie>
                <Tooltip formatter={(value, name) => [`$${value}M`, name]} />
                <Legend />
              </PieChart>
            </ResponsiveContainer>
            
            {/* Chart Values Input */}
            <div className="mt-4 grid grid-cols-3 gap-3">
              <div className="space-y-1">
                <Label htmlFor="tam-value" className="text-xs flex items-center gap-1">
                  <div className="w-3 h-3 rounded-full bg-blue-500"></div>
                  TAM ($M)
                </Label>
                <Input
                  id="tam-value"
                  type="number"
                  value={marketData.tamValue}
                  onChange={(e) => updateMarketData('tamValue', Number(e.target.value))}
                  className="text-xs"
                  data-testid="tam-value-input"
                />
              </div>
              <div className="space-y-1">
                <Label htmlFor="sam-value" className="text-xs flex items-center gap-1">
                  <div className="w-3 h-3 rounded-full bg-emerald-500"></div>
                  SAM ($M)
                </Label>
                <Input
                  id="sam-value"
                  type="number"
                  value={marketData.samValue}
                  onChange={(e) => updateMarketData('samValue', Number(e.target.value))}
                  className="text-xs"
                  data-testid="sam-value-input"
                />
              </div>
              <div className="space-y-1">
                <Label htmlFor="som-value" className="text-xs flex items-center gap-1">
                  <div className="w-3 h-3 rounded-full bg-violet-500"></div>
                  SOM ($M)
                </Label>
                <Input
                  id="som-value"
                  type="number"
                  value={marketData.somValue}
                  onChange={(e) => updateMarketData('somValue', Number(e.target.value))}
                  className="text-xs"
                  data-testid="som-value-input"
                />
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Form Fields */}
      <div className="grid gap-6">
        {/* TAM */}
        <Card data-testid="tam-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <Target className="w-4 h-4 text-blue-500" />
              TAM (Total Addressable Market)
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="tam-input">The entire demand for your product or service</Label>
              <Textarea
                id="tam-input"
                placeholder="e.g., 'The global project management software market is valued at $6.68 billion annually, with an expected CAGR of 10.67% through 2028.'"
                value={marketData.tam}
                onChange={(e) => updateMarketData('tam', e.target.value)}
                rows={3}
                data-testid="tam-textarea"
              />
              <p className="text-xs text-muted-foreground">
                Include market size, growth rates, and geographic scope
              </p>
            </div>
          </CardContent>
        </Card>

        {/* SAM */}
        <Card data-testid="sam-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <Zap className="w-4 h-4 text-green-500" />
              SAM (Serviceable Available Market)
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="sam-input">The portion of TAM you can actually reach</Label>
              <Textarea
                id="sam-input"
                placeholder="e.g., 'Small to medium businesses (10-500 employees) in North America using cloud-based tools, representing $2.1 billion of the total market.'"
                value={marketData.sam}
                onChange={(e) => updateMarketData('sam', e.target.value)}
                rows={3}
                data-testid="sam-textarea"
              />
              <p className="text-xs text-muted-foreground">
                Factor in geographic limitations, product fit, and distribution channels
              </p>
            </div>
          </CardContent>
        </Card>

        {/* SOM */}
        <Card data-testid="som-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <Target className="w-4 h-4 text-purple-500" />
              SOM (Serviceable Obtainable Market)
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="som-input">The market share you can realistically capture</Label>
              <Textarea
                id="som-input"
                placeholder="e.g., 'Targeting 0.5% market share within 3 years, representing $10.5M in annual revenue based on competitive positioning and go-to-market strategy.'"
                value={marketData.som}
                onChange={(e) => updateMarketData('som', e.target.value)}
                rows={3}
                data-testid="som-textarea"
              />
              <p className="text-xs text-muted-foreground">
                Consider competitive landscape, resources, and realistic growth projections
              </p>
            </div>
          </CardContent>
        </Card>

        {/* Sources */}
        <Card data-testid="sources-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <FileText className="w-4 h-4 text-orange-500" />
              Sources & Assumptions
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="sources-input">Data sources and key assumptions</Label>
              <Textarea
                id="sources-input"
                placeholder="e.g., 'Data from Gartner Research 2024, Grand View Research, internal customer surveys. Assumes 15% annual growth in SMB cloud adoption.'"
                value={marketData.sources}
                onChange={(e) => updateMarketData('sources', e.target.value)}
                rows={3}
                data-testid="sources-textarea"
              />
              <p className="text-xs text-muted-foreground">
                Include research sources, surveys, and key assumptions made in calculations
              </p>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Instructions */}
      <Card className="border-dashed">
        <CardContent className="pt-4">
          <div className="text-sm text-muted-foreground space-y-2">
            <p><strong>Market sizing framework:</strong></p>
            <ul className="list-disc list-inside space-y-1 ml-2">
              <li><strong>TAM:</strong> Start with the broadest possible market size</li>
              <li><strong>SAM:</strong> Narrow down based on your product's actual reach</li>
              <li><strong>SOM:</strong> Be realistic about what you can achieve with your resources</li>
              <li>Use the chart above to visualize relative market sizes</li>
              <li>Use credible sources like industry reports, government data, or primary research</li>
              <li>Update your market sizing regularly as you gather more data</li>
            </ul>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}