import { useState, useRef } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { usePlanStoreV2 } from "@/stores/usePlanStore.v2";
import { useBrandKitStore } from "@/stores/useBrandKitStore";
import { getBrandPalette } from "@/lib/brand";
import { ChartSnapshotButton } from "@/components/ChartSnapshotButton";
import { 
  LineChart, 
  Line, 
  AreaChart, 
  Area, 
  BarChart, 
  Bar, 
  PieChart,
  Pie,
  Cell,
  XAxis, 
  YAxis, 
  CartesianGrid, 
  Tooltip, 
  ResponsiveContainer,
  Legend
} from "recharts";
import { Upload, FileText, AlertCircle, CheckCircle, BarChart3, Table, Download } from "lucide-react";
import Papa from "papaparse";
import * as XLSX from "xlsx";

interface DataRow {
  [key: string]: string | number;
}

interface ParseResult {
  success: boolean;
  message: string;
  data?: DataRow[];
  headers?: string[];
}

const CHART_TYPES = [
  { value: "line", label: "Line Chart", icon: "📈" },
  { value: "bar", label: "Bar Chart", icon: "📊" },
  { value: "pie", label: "Pie Chart", icon: "🥧" },
] as const;

export function CustomChartBuilder({ 
  targetSectionId 
}: { 
  targetSectionId: string 
}) {
  const [title, setTitle] = useState("Custom Chart");
  const [chartType, setChartType] = useState<"line" | "bar" | "pie">("line");
  const [data, setData] = useState<DataRow[]>([]);
  const [headers, setHeaders] = useState<string[]>([]);
  const [xKey, setXKey] = useState<string>("");
  const [yKey, setYKey] = useState<string>("");
  const [parseResult, setParseResult] = useState<ParseResult | null>(null);
  const [isProcessing, setIsProcessing] = useState(false);
  
  const update = usePlanStoreV2((s) => s.updateSection);
  const chartRef = useRef<HTMLDivElement>(null);
  const brand = useBrandKitStore((s) => s.getCurrentKit());
  
  // Get brand-aware color palette
  const palette = getBrandPalette(brand);

  const handleFileUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    setIsProcessing(true);
    setParseResult(null);

    // Check file type
    const isCSV = file.name.toLowerCase().endsWith('.csv');
    const isExcel = file.name.toLowerCase().endsWith('.xlsx') || file.name.toLowerCase().endsWith('.xls');

    if (isCSV) {
      // Parse CSV file
      Papa.parse(file, {
        header: true,
        skipEmptyLines: true,
        complete: (results) => {
          try {
            if (results.errors.length > 0) {
              throw new Error(`CSV parsing error: ${results.errors[0].message}`);
            }

            const parsedData = results.data as DataRow[];
            const dataHeaders = results.meta.fields || [];

            if (parsedData.length === 0) {
              throw new Error("No data found in CSV file");
            }

            setData(parsedData);
            setHeaders(dataHeaders);
            setXKey(dataHeaders[0] || "");
            setYKey(dataHeaders[1] || "");

            setParseResult({
              success: true,
              message: `Successfully imported ${parsedData.length} rows from CSV`,
              data: parsedData,
              headers: dataHeaders,
            });
          } catch (error) {
            setParseResult({
              success: false,
              message: error instanceof Error ? error.message : "Import failed",
            });
          } finally {
            setIsProcessing(false);
            event.target.value = '';
          }
        },
        error: (error) => {
          setParseResult({
            success: false,
            message: `File parsing error: ${error.message}`,
          });
          setIsProcessing(false);
          event.target.value = '';
        }
      });
    } else if (isExcel) {
      // Parse Excel file
      const reader = new FileReader();
      reader.onload = (e) => {
        try {
          const data = new Uint8Array(e.target?.result as ArrayBuffer);
          const workbook = XLSX.read(data, { type: 'array' });
          const sheetName = workbook.SheetNames[0];
          const worksheet = workbook.Sheets[sheetName];
          const jsonData = XLSX.utils.sheet_to_json(worksheet, { header: 1 });

          if (jsonData.length === 0) {
            throw new Error("No data found in Excel file");
          }

          // First row as headers
          const dataHeaders = (jsonData[0] as string[]).map(h => String(h));
          const rows = jsonData.slice(1).map((row: any) => {
            const obj: DataRow = {};
            dataHeaders.forEach((header, index) => {
              obj[header] = row[index] || "";
            });
            return obj;
          }).filter(row => Object.values(row).some(val => val !== ""));

          setData(rows);
          setHeaders(dataHeaders);
          setXKey(dataHeaders[0] || "");
          setYKey(dataHeaders[1] || "");

          setParseResult({
            success: true,
            message: `Successfully imported ${rows.length} rows from Excel`,
            data: rows,
            headers: dataHeaders,
          });
        } catch (error) {
          setParseResult({
            success: false,
            message: error instanceof Error ? error.message : "Excel parsing failed",
          });
        } finally {
          setIsProcessing(false);
          event.target.value = '';
        }
      };
      reader.readAsArrayBuffer(file);
    } else {
      setParseResult({
        success: false,
        message: "Unsupported file type. Please upload a CSV or Excel file.",
      });
      setIsProcessing(false);
      event.target.value = '';
    }
  };

  const insertAsTable = () => {
    if (data.length === 0) return;

    // Create markdown table
    const tableHeaders = headers.map(h => `| ${h} `).join('') + '|';
    const tableSeparator = headers.map(() => '| --- ').join('') + '|';
    const tableRows = data.slice(0, 20).map(row => 
      headers.map(h => `| ${row[h] || ''} `).join('') + '|'
    ).join('\n');

    const markdownTable = `### ${title}\n\n${tableHeaders}\n${tableSeparator}\n${tableRows}\n\n*Showing first 20 rows of ${data.length} total rows*`;

    update(targetSectionId, (prev: any) => ({
      content: `${prev?.content ? prev.content + "\n\n" : ""}${markdownTable}`,
    }));
  };

  const formatChartData = () => {
    if (!xKey || !yKey || data.length === 0) return [];
    
    return data.map(row => ({
      [xKey]: row[xKey],
      [yKey]: chartType === "pie" ? Number(row[yKey]) || 0 : row[yKey],
      value: Number(row[yKey]) || 0, // For pie charts
    }));
  };

  const renderChart = () => {
    const chartData = formatChartData();
    if (chartData.length === 0) return null;

    const commonProps = {
      width: "100%",
      height: 300,
    };

    switch (chartType) {
      case "line":
        return (
          <div style={{ fontFamily: brand?.fontBody || "Inter, ui-sans-serif" }}>
          <ResponsiveContainer {...commonProps}>
            <LineChart data={chartData}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey={xKey} tick={{ fontSize: 12 }} />
              <YAxis tick={{ fontSize: 12 }} />
              <Tooltip />
              <Line 
                type="monotone" 
                dataKey={yKey} 
                stroke={palette[0]} 
                strokeWidth={2}
                dot={{ fill: palette[0], strokeWidth: 2, r: 4 }}
              />
            </LineChart>
          </ResponsiveContainer>
          </div>
        );

      case "bar":
        return (
          <div style={{ fontFamily: brand?.fontBody || "Inter, ui-sans-serif" }}>
          <ResponsiveContainer {...commonProps}>
            <BarChart data={chartData}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey={xKey} tick={{ fontSize: 12 }} />
              <YAxis tick={{ fontSize: 12 }} />
              <Tooltip />
              <Bar dataKey={yKey} fill={palette[0]} radius={[2, 2, 0, 0]} />
            </BarChart>
          </ResponsiveContainer>
          </div>
        );

      case "pie":
        return (
          <div style={{ fontFamily: brand?.fontBody || "Inter, ui-sans-serif" }}>
          <ResponsiveContainer {...commonProps}>
            <PieChart>
              <Pie
                data={chartData}
                cx="50%"
                cy="50%"
                outerRadius={80}
                fill={palette[2]}
                dataKey="value"
                label={({ name, percent }) => `${name}: ${(percent * 100).toFixed(0)}%`}
              >
                {chartData.map((_, index) => (
                  <Cell key={`cell-${index}`} fill={palette[index % palette.length]} />
                ))}
              </Pie>
              <Tooltip />
            </PieChart>
          </ResponsiveContainer>
          </div>
        );

      default:
        return null;
    }
  };

  return (
    <Card className="w-full" data-testid="custom-chart-builder">
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <BarChart3 className="w-5 h-5" />
          Custom Chart Builder
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-6">
        {/* Chart Configuration */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div className="space-y-2">
            <Label htmlFor="chart-title">Chart Title</Label>
            <Input
              id="chart-title"
              value={title}
              onChange={(e) => setTitle(e.target.value)}
              placeholder="Enter chart title"
              data-testid="input-chart-title"
            />
          </div>
          <div className="space-y-2">
            <Label>Chart Type</Label>
            <Select value={chartType} onValueChange={(value) => setChartType(value as "line" | "bar" | "pie")}>
              <SelectTrigger data-testid="select-chart-type">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                {CHART_TYPES.map((type) => (
                  <SelectItem key={type.value} value={type.value}>
                    {type.icon} {type.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>

        {/* File Upload */}
        <div className="space-y-3">
          <div className="space-y-2">
            <Label htmlFor="data-file" className="text-sm font-medium">
              Import Data File
            </Label>
            <div className="text-xs text-muted-foreground">
              <p>Supported formats: CSV (.csv), Excel (.xlsx, .xls)</p>
            </div>
          </div>
          
          <div className="flex items-center gap-3">
            <Input
              id="data-file"
              type="file"
              accept=".csv,.xlsx,.xls"
              onChange={handleFileUpload}
              disabled={isProcessing}
              className="flex-1"
              data-testid="input-data-file"
            />
            <FileText className="w-4 h-4 text-muted-foreground" />
          </div>

          {parseResult && (
            <div className={`flex items-center gap-2 text-sm p-2 rounded-lg ${
              parseResult.success 
                ? 'bg-green-50 text-green-700 border border-green-200' 
                : 'bg-red-50 text-red-700 border border-red-200'
            }`} data-testid="parse-result">
              {parseResult.success ? (
                <CheckCircle className="w-4 h-4" />
              ) : (
                <AlertCircle className="w-4 h-4" />
              )}
              <span>{parseResult.message}</span>
            </div>
          )}
        </div>

        {/* Data Mapping - Only show if we have data */}
        {data.length > 0 && headers.length > 0 && (
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label>X-Axis (Category)</Label>
              <Select value={xKey} onValueChange={setXKey}>
                <SelectTrigger data-testid="select-x-key">
                  <SelectValue placeholder="Select X-axis column" />
                </SelectTrigger>
                <SelectContent>
                  {headers.map((header) => (
                    <SelectItem key={header} value={header}>
                      {header}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Y-Axis (Value)</Label>
              <Select value={yKey} onValueChange={setYKey}>
                <SelectTrigger data-testid="select-y-key">
                  <SelectValue placeholder="Select Y-axis column" />
                </SelectTrigger>
                <SelectContent>
                  {headers.map((header) => (
                    <SelectItem key={header} value={header}>
                      {header}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
        )}

        {/* Data Preview Table - Only show if we have data */}
        {data.length > 0 && (
          <div className="space-y-3">
            <Label className="text-sm font-medium">Data Preview (First 10 rows)</Label>
            <div className="border rounded-lg overflow-x-auto" data-testid="data-preview">
              <table className="w-full text-sm">
                <thead className="bg-muted/50">
                  <tr>
                    {headers.map((header) => (
                      <th key={header} className="px-3 py-2 text-left font-medium">
                        {header}
                      </th>
                    ))}
                  </tr>
                </thead>
                <tbody>
                  {data.slice(0, 10).map((row, index) => (
                    <tr key={index} className="border-t">
                      {headers.map((header) => (
                        <td key={header} className="px-3 py-2">
                          {String(row[header] || "")}
                        </td>
                      ))}
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
            {data.length > 10 && (
              <p className="text-xs text-muted-foreground">
                Showing 10 of {data.length} total rows
              </p>
            )}
          </div>
        )}

        {/* Chart Preview - Only show if we have data and keys selected */}
        {data.length > 0 && xKey && yKey && (
          <div className="space-y-3">
            <div className="flex items-center justify-between">
              <Label className="text-sm font-medium">Chart Preview</Label>
              <ChartSnapshotButton 
                targetSectionId={targetSectionId} 
                getElement={() => chartRef.current} 
                title={title} 
              />
            </div>
            <Card>
              <CardHeader>
                <CardTitle className="text-lg">{title}</CardTitle>
              </CardHeader>
              <CardContent>
                <div ref={chartRef} data-testid="chart-preview">
                  {renderChart()}
                </div>
              </CardContent>
            </Card>
          </div>
        )}

        {/* Export Options - Only show if we have data */}
        {data.length > 0 && (
          <div className="flex gap-3 pt-4 border-t">
            <Button
              onClick={insertAsTable}
              variant="outline"
              className="flex items-center gap-2"
              data-testid="button-insert-table"
            >
              <Table className="w-4 h-4" />
              Insert as Table
            </Button>
            {xKey && yKey && (
              <ChartSnapshotButton 
                targetSectionId={targetSectionId} 
                getElement={() => chartRef.current} 
                title={title} 
              />
            )}
          </div>
        )}

        {/* Processing Indicator */}
        {isProcessing && (
          <div className="text-center py-4">
            <div className="inline-flex items-center gap-2 text-sm text-muted-foreground">
              <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-primary"></div>
              Processing file...
            </div>
          </div>
        )}

        {/* Sample Templates Download */}
        <div className="border-t pt-4">
          <div className="space-y-2">
            <Label className="text-sm font-medium">Sample Data Templates</Label>
            <div className="flex gap-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => {
                  const csvContent = "Month,Revenue,Expenses\nJan,5000,3000\nFeb,5500,3200\nMar,6000,3500\nApr,6500,3800\nMay,7000,4000\nJun,7500,4200";
                  const blob = new Blob([csvContent], { type: 'text/csv' });
                  const url = URL.createObjectURL(blob);
                  const a = document.createElement('a');
                  a.href = url;
                  a.download = 'sample-chart-data.csv';
                  a.click();
                  URL.revokeObjectURL(url);
                }}
                data-testid="button-download-sample"
              >
                <Download className="w-4 h-4 mr-2" />
                Download Sample CSV
              </Button>
            </div>
          </div>
        </div>
      </CardContent>
    </Card>
  );
}