import React, { useState } from "react";
import { useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Download, Wand2 } from "lucide-react";
import { PaywallModal } from "@/components/PaywallModal";
import { useUserPlan } from "@/hooks/useUserPlan";
import { PreviewLightbox } from "@/components/common/PreviewLightbox";

type Props = {
  id: string;
  name: string;
  description?: string;
  tags?: string[];
  previewUrl?: string;
  source?: "firebase" | "local";
  typeOverride?: string; // e.g., "SVG" or "PNG" (optional)
  onClick?: () => void;
  defaults?: Record<string, string>; // Default field values for template
};

export function TemplateCard({
  id,
  name,
  description,
  tags = [],
  previewUrl,
  source = "firebase",
  typeOverride,
  onClick,
  defaults,
}: Props) {
  const [, setLocation] = useLocation();
  const [showPaywall, setShowPaywall] = useState(false);
  const [showPreview, setShowPreview] = useState(false);
  const [hover, setHover] = useState(false);
  const { isPremium } = useUserPlan() ?? { isPremium: false };
  
  // Derive a type badge if caller didn't pass one
  const typeFromTags = tags.find(t => t && ["SVG", "PNG", "JPG", "JPEG"].includes(t.toUpperCase()));
  const typeBadge = (typeOverride || typeFromTags || "").toUpperCase();

  const handleDownload = async () => {
    try {
      if (previewUrl) {
        // Download from URL
        const response = await fetch(previewUrl);
        const blob = await response.blob();
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `${id}.svg`;
        a.click();
        URL.revokeObjectURL(url);
      } else {
        console.warn('[TemplateCard] No preview URL found for download', id);
      }
    } catch (error) {
      console.error('[TemplateCard] Download failed', error);
    }
  };

  const handleCustomize = () => {
    if (!isPremium) {
      setShowPaywall(true);
      return;
    }
    
    // Store template data in sessionStorage for logo composer
    const templateData = {
      templateId: id,
      name,
      svgUrl: previewUrl?.endsWith(".svg") ? previewUrl : undefined,
      rasterUrl: !previewUrl?.endsWith(".svg") ? previewUrl : undefined,
      defaults,
    };
    
    sessionStorage.setItem('templateData', JSON.stringify(templateData));
    setLocation(`/brand-development/ai-logo-creator/logo-composer-${id}`);
  };

  return (
    <>
      <PaywallModal
        isOpen={showPaywall}
        onClose={() => setShowPaywall(false)}
        feature="logo customization"
        description="Customize templates with text, colors, and shapes in the Logo Composer"
      />

      <PreviewLightbox
        open={showPreview}
        onClose={() => setShowPreview(false)}
        title={name}
        imgSrc={previewUrl || ""}
        onDownload={handleDownload}
        onCustomize={handleCustomize}
        customizeDisabled={false}
      />
      
      <div 
        className="relative rounded-2xl shadow border border-gray-200 bg-white overflow-hidden group transition-shadow duration-200 hover:shadow-lg hover:border-gray-300"
        onMouseEnter={() => setHover(true)}
        onMouseLeave={() => setHover(false)}
      >
        {/* Preview */}
        <div 
          className="relative w-full aspect-[4/3] bg-gray-50 border-b overflow-hidden cursor-zoom-in"
          onClick={() => previewUrl && setShowPreview(true)}
          data-testid={`preview-thumbnail-${id}`}
        >
          {previewUrl ? (
            <img 
              src={previewUrl} 
              alt={name} 
              className="w-full h-full object-contain transition-transform duration-300 ease-out group-hover:scale-[1.06] motion-reduce:transition-none motion-reduce:transform-none" 
              loading="lazy"
              draggable={false}
            />
          ) : (
            <div className="absolute inset-0 flex items-center justify-center text-gray-400 text-sm">No preview</div>
          )}
        </div>

        {/* Desktop Hover Overlay */}
        <div 
          className={`absolute inset-0 bg-black/60 flex-col items-center justify-center gap-3 transition-opacity duration-200 ease-out hidden md:flex pointer-events-none ${
            hover ? "opacity-100" : "opacity-0"
          }`}
        >
          <Button
            onClick={handleDownload}
            variant="outline"
            size="default"
            className="bg-white hover:bg-gray-50 text-black hover:text-black pointer-events-auto"
            aria-label="Download this logo as SVG"
            data-testid={`button-download-${id}`}
          >
            <Download className="h-4 w-4 mr-2" />
            Download SVG
          </Button>
          <Button
            onClick={handleCustomize}
            size="default"
            className="bg-[#FF8B00] hover:bg-[#e67d00] text-white pointer-events-auto"
            aria-label="Open this logo in the Composer"
            data-testid={`button-customize-${id}`}
          >
            <Wand2 className="h-4 w-4 mr-2" />
            Customize in Composer
          </Button>
        </div>

        {/* Info */}
        <div className="p-4 space-y-3">
          <div className="flex items-start justify-between gap-3">
            <div className="min-w-0">
              <h3 className="text-lg font-semibold truncate" title={name}>
                {name}
              </h3>
              <p className="text-sm text-gray-600 line-clamp-2" title={description}>
                {description || "—"}
              </p>
            </div>

            {/* Type badge (SVG / PNG / JPG) */}
            {typeBadge && (
              <span className="text-xs px-2 py-1 rounded-full border bg-gray-50 shrink-0">
                {typeBadge}
              </span>
            )}
          </div>

          {/* Footer tags (optional) */}
          {tags.length > 0 && (
            <div className="flex flex-wrap gap-2">
              {tags.map((t) => (
                <span key={t} className="text-[11px] px-2 py-0.5 rounded-full bg-gray-100 text-gray-700">
                  {t}
                </span>
              ))}
            </div>
          )}

          {/* Mobile: Always visible buttons */}
          <div className="flex md:hidden flex-col gap-2 pt-2 border-t">
            <Button
              onClick={handleDownload}
              variant="outline"
              size="sm"
              className="w-full text-black hover:text-black"
              aria-label="Download this logo as SVG"
              data-testid={`button-download-mobile-${id}`}
            >
              <Download className="h-4 w-4 mr-2" />
              Download SVG
            </Button>
            <Button
              onClick={handleCustomize}
              size="sm"
              className="w-full bg-[#FF8B00] hover:bg-[#e67d00] text-white"
              aria-label="Open this logo in the Composer"
              data-testid={`button-customize-mobile-${id}`}
            >
              <Wand2 className="h-4 w-4 mr-2" />
              Customize in Composer
            </Button>
          </div>
        </div>
      </div>
    </>
  );
}