import { useState, useEffect } from "react";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Loader2, Sparkles, X, Download, RotateCcw } from "lucide-react";
import { generateLogoSvg } from "@/services/logo-ai";
import { sanitizeAndNormalizeSvg, isValidSvg } from "@/lib/svg-sanitize";
import { useToast } from "@/hooks/use-toast";
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from "@/components/ui/collapsible";
import { ChevronDown } from "lucide-react";

interface LogoAIGeneratorProps {
  brandName?: string;
  onInsert?: (svg: string) => void;
  onReplace?: (svg: string) => void;
  hasExistingIcon?: boolean;
  defaultOpen?: boolean;
}

export function LogoAIGenerator({
  brandName,
  onInsert,
  onReplace,
  hasExistingIcon = false,
  defaultOpen = false
}: LogoAIGeneratorProps) {
  const { toast } = useToast();
  const [isOpen, setIsOpen] = useState(defaultOpen);
  const [prompt, setPrompt] = useState("");
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [svgPreview, setSvgPreview] = useState<string | null>(null);
  const [svgBlobUrl, setSvgBlobUrl] = useState<string | null>(null);

  // Cleanup blob URL on unmount or when SVG changes
  useEffect(() => {
    return () => {
      if (svgBlobUrl) {
        URL.revokeObjectURL(svgBlobUrl);
      }
    };
  }, [svgBlobUrl]);

  const handleGenerate = async () => {
    if (!prompt.trim()) {
      setError("Please enter a prompt description");
      return;
    }

    if (prompt.length > 400) {
      setError("Prompt must be less than 400 characters");
      return;
    }

    setLoading(true);
    setError(null);
    setSvgPreview(null);
    
    // Cleanup old blob URL
    if (svgBlobUrl) {
      URL.revokeObjectURL(svgBlobUrl);
      setSvgBlobUrl(null);
    }

    try {
      // Generate logo
      const result = await generateLogoSvg({
        prompt: prompt.trim(),
        brandName
      });

      // Defensive: Check for empty/invalid SVG
      if (!result.svg || result.svg.length < 100) {
        throw new Error("Generation returned an empty or invalid SVG. Please try again.");
      }

      // Validate SVG
      if (!isValidSvg(result.svg)) {
        throw new Error("Generated SVG is invalid");
      }

      // Sanitize and normalize
      const { svg: cleanSvg, warnings } = sanitizeAndNormalizeSvg(result.svg);

      // Show warnings if any
      if (warnings.length > 0) {
        console.log('[logo-ai] Sanitization warnings:', warnings);
        const hasTextWarning = warnings.some(w => w.includes('text'));
        if (hasTextWarning) {
          toast({
            title: "Text Removed",
            description: "This generator produces icon-only SVGs; text was removed.",
            variant: "default"
          });
        }
      }

      // Store SVG with transparent background and create blob URL for preview
      let transparentSvg = cleanSvg;
      // Add background:none to SVG - handle both with and without existing style attribute
      if (/<svg[^>]*\sstyle=/i.test(transparentSvg)) {
        // SVG already has style attribute - append to it
        transparentSvg = transparentSvg.replace(
          /(<svg[^>]*\sstyle=["'])([^"']*)(["'])/i,
          '$1$2;background:none$3'
        );
      } else {
        // SVG has no style attribute - add one
        transparentSvg = transparentSvg.replace(/<svg\b/i, '<svg style="background:none"');
      }
      setSvgPreview(transparentSvg);
      
      try {
        const blob = new Blob([transparentSvg], { type: 'image/svg+xml' });
        const url = URL.createObjectURL(blob);
        setSvgBlobUrl(url);
      } catch (blobError) {
        console.warn('[logo-ai] Blob URL creation failed, using data URL fallback');
        // Fallback to data URL if blob fails
        const dataUrl = `data:image/svg+xml;utf8,${encodeURIComponent(transparentSvg)}`;
        setSvgBlobUrl(dataUrl);
      }

      console.log('[logo-ai] Logo generated and sanitized successfully');

    } catch (err) {
      const message = err instanceof Error ? err.message : "Couldn't generate a logo right now. Try again.";
      setError(message);
      toast({
        title: "Generation Failed",
        description: message,
        variant: "destructive"
      });
      console.error('[logo-ai] Generation error:', err);
    } finally {
      setLoading(false);
    }
  };

  const handleInsert = () => {
    if (!svgPreview) return;
    onInsert?.(svgPreview);
    toast({
      title: "Logo Inserted",
      description: "AI-generated icon added to canvas",
      variant: "default"
    });
  };

  const handleReplace = () => {
    if (!svgPreview) return;
    onReplace?.(svgPreview);
    toast({
      title: "Icon Replaced",
      description: "Existing icon updated with new design",
      variant: "default"
    });
  };

  const handleDiscard = () => {
    if (svgBlobUrl) {
      URL.revokeObjectURL(svgBlobUrl);
    }
    setSvgPreview(null);
    setSvgBlobUrl(null);
    setPrompt("");
    setError(null);
  };

  const handleDownloadSVG = () => {
    if (!svgPreview) return;
    
    const blob = new Blob([svgPreview], { type: 'image/svg+xml' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.download = `ai-logo-${Date.now()}.svg`;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);

    toast({
      title: "Downloaded",
      description: "SVG logo saved successfully",
      variant: "default"
    });
  };

  const handleDownloadPNG = async () => {
    if (!svgPreview) return;

    try {
      // Create an image from SVG
      const img = new Image();
      const svgBlob = new Blob([svgPreview], { type: 'image/svg+xml;charset=utf-8' });
      const url = URL.createObjectURL(svgBlob);

      img.onload = () => {
        // Create canvas and draw image
        const canvas = document.createElement('canvas');
        canvas.width = img.width || 512;
        canvas.height = img.height || 512;
        const ctx = canvas.getContext('2d');
        
        if (ctx) {
          // Ensure transparent background - don't paint white rect
          ctx.clearRect(0, 0, canvas.width, canvas.height);
          ctx.drawImage(img, 0, 0);
          
          // Convert to PNG and download
          canvas.toBlob((blob) => {
            if (blob) {
              const pngUrl = URL.createObjectURL(blob);
              const link = document.createElement('a');
              link.href = pngUrl;
              link.download = `ai-logo-${Date.now()}.png`;
              document.body.appendChild(link);
              link.click();
              document.body.removeChild(link);
              URL.revokeObjectURL(pngUrl);

              toast({
                title: "Downloaded",
                description: "PNG logo saved successfully",
                variant: "default"
              });
            }
          }, 'image/png');
        }
        
        URL.revokeObjectURL(url);
      };

      img.onerror = () => {
        URL.revokeObjectURL(url);
        toast({
          title: "Download Failed",
          description: "Could not convert to PNG",
          variant: "destructive"
        });
      };

      img.src = url;
    } catch (err) {
      toast({
        title: "Download Failed",
        description: "Could not generate PNG",
        variant: "destructive"
      });
    }
  };

  return (
    <Collapsible open={isOpen} onOpenChange={setIsOpen} className="w-full">
      <CollapsibleTrigger className="w-full">
        <div className="flex items-center justify-between w-full px-4 py-3 bg-white dark:bg-gray-900 rounded-lg border border-gray-200 dark:border-gray-700 hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors">
          <div className="flex items-center gap-2">
            <Sparkles className="w-4 h-4 text-emerald-600" />
            <span className="font-medium text-sm text-gray-900 dark:text-gray-100">
              Generate AI-powered SVG logo
            </span>
          </div>
          <ChevronDown 
            className={`w-4 h-4 text-gray-500 transition-transform ${isOpen ? 'rotate-180' : ''}`} 
          />
        </div>
      </CollapsibleTrigger>

      <CollapsibleContent className="mt-4 space-y-4">
        {/* Prompt Input */}
        <div className="space-y-2">
          <Textarea
            value={prompt}
            onChange={(e) => setPrompt(e.target.value)}
            placeholder="Describe your logo (minimal icon, geometric, vector, no text)"
            className="min-h-[80px] resize-none"
            disabled={loading}
            data-testid="input-logo-prompt"
          />
          <div className="flex justify-between items-center">
            <span className="text-xs text-gray-500">
              {prompt.length}/400 characters
            </span>
            <Button
              onClick={handleGenerate}
              disabled={loading || !prompt.trim() || prompt.length > 400}
              className="bg-emerald-600 hover:bg-emerald-700 text-white"
              data-testid="button-generate-logo"
            >
              {loading ? (
                <>
                  <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  Generating...
                </>
              ) : (
                <>
                  <Sparkles className="w-4 h-4 mr-2" />
                  Generate Logo
                </>
              )}
            </Button>
          </div>
        </div>

        {/* Loading Message */}
        {loading && (
          <div className="text-sm text-gray-600 dark:text-gray-400 text-center">
            This may take up to 3 minutes...
          </div>
        )}

        {/* Error Message */}
        {error && (
          <div 
            className="rounded-lg border border-red-300 bg-red-50 dark:bg-red-900/20 dark:border-red-800 p-3 text-sm text-red-700 dark:text-red-400"
            data-testid="error-message"
          >
            <strong>Error:</strong> {error}
          </div>
        )}

        {/* Preview Box */}
        <div className="rounded-xl border border-gray-200 dark:border-gray-700 overflow-hidden" style={{
          backgroundImage: 'url("data:image/svg+xml,%3csvg width=\'20\' height=\'20\' xmlns=\'http://www.w3.org/2000/svg\'%3e%3cdefs%3e%3cpattern id=\'checker\' patternUnits=\'userSpaceOnUse\' width=\'20\' height=\'20\'%3e%3crect x=\'0\' y=\'0\' width=\'10\' height=\'10\' fill=\'%23f3f4f6\'/%3e%3crect x=\'10\' y=\'10\' width=\'10\' height=\'10\' fill=\'%23f3f4f6\'/%3e%3c/pattern%3e%3c/defs%3e%3crect width=\'100%25\' height=\'100%25\' fill=\'url(%23checker)\'/%3e%3c/svg%3e")',
          backgroundSize: '20px 20px'
        }}>
          <div className="px-4 py-3 border-b border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-900">
            <span className="text-sm font-medium text-gray-700 dark:text-gray-300">Preview</span>
          </div>
          
          <div
            className="flex items-center justify-center relative"
            style={{
              minHeight: "260px",
              maxHeight: "300px",
              overflow: 'auto'
            }}
            data-testid="logo-preview"
          >
            {loading && (
              <Loader2 className="w-8 h-8 animate-spin text-emerald-600" />
            )}
            
            {svgPreview && svgBlobUrl && !loading && (
              <img
                src={svgBlobUrl}
                alt="Generated AI logo"
                className="p-4"
                style={{ display: 'block', maxWidth: '280px', maxHeight: '280px', objectFit: 'contain', background: 'transparent' }}
                data-testid="generated-logo-preview"
              />
            )}
          </div>
        </div>

        {/* Action Buttons - only show when preview is ready */}
        {svgPreview && !loading && (
          <div className="flex flex-wrap gap-2">
            {/* Download buttons - always show */}
            <Button
              onClick={handleDownloadSVG}
              className="bg-emerald-600 hover:bg-emerald-700 text-white"
              data-testid="button-download-svg"
            >
              <Download className="w-4 h-4 mr-2" />
              Download SVG
            </Button>
            
            <Button
              onClick={handleDownloadPNG}
              className="bg-[#FF8B00] text-white hover:bg-orange-600"
              data-testid="button-download-png"
            >
              <Download className="w-4 h-4 mr-2" />
              Download PNG
            </Button>
            
            {/* Canvas insert buttons - only show if callbacks provided */}
            {onInsert && (
              <div className="flex flex-col">
                <Button
                  onClick={handleInsert}
                  variant="outline"
                  className="bg-white text-black border-2 border-[#FF8B00] hover:bg-orange-50 hover:text-black font-normal"
                  aria-label="Customize Logo"
                  title="Customize Logo"
                  data-testid="button-insert-canvas"
                >
                  Customize Logo
                </Button>
                <div className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                  Step 2: Fine-tune text, shapes & colors →
                </div>
              </div>
            )}
            
            {hasExistingIcon && onReplace && (
              <Button
                onClick={handleReplace}
                variant="ghost"
                className="text-gray-700 dark:text-gray-300"
                data-testid="button-replace-icon"
              >
                Replace Existing Icon
              </Button>
            )}
            
            <Button
              onClick={handleDiscard}
              variant="outline"
              className="bg-white text-black border-2 border-[#FF8B00] hover:bg-orange-50 ml-auto"
              data-testid="button-reset"
            >
              <RotateCcw className="w-4 h-4 mr-1" />
              Reset
            </Button>
          </div>
        )}
      </CollapsibleContent>
    </Collapsible>
  );
}
