import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { useFinanceStoreV2 } from "@/stores/useFinanceStore.v2";
import { usePlanStoreV2 } from "@/stores/usePlanStore.v2";
import { FinanceInputs } from "@/components/finance/FinanceInputs";
import { ExpensesTable } from "@/components/finance/ExpensesTable";
import { FinanceCharts } from "@/components/finance/FinanceCharts";
import { CsvImport } from "@/components/finance/CsvImport";
import { CustomChartBuilder } from "@/components/visuals/CustomChartBuilder";
import { FinanceInsightsAI } from "@/components/finance/FinanceInsightsAI";
import { FinanceSidebar } from "@/components/finance/FinanceSidebar";
import { Calculator, BarChart3, Upload, Brain, Settings, PieChart } from "lucide-react";
import { useEffect } from "react";

export function FinancialsPanel() {
  const { recompute } = useFinanceStoreV2();
  const { sections } = usePlanStoreV2();
  
  // Find the Financial Plan section for CustomChartBuilder
  const financialSection = sections.find(s => s.kind === "financials");
  const financialSectionId = financialSection?.id || "";

  // Auto-recompute on mount to ensure data is fresh
  useEffect(() => {
    const timer = setTimeout(recompute, 100);
    return () => clearTimeout(timer);
  }, [recompute]);

  return (
    <div className="space-y-6" data-testid="financials-panel">
      {/* Header */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Calculator className="w-6 h-6" />
            Financial Planning & Analysis V2
          </CardTitle>
          <p className="text-sm text-muted-foreground">
            Comprehensive financial modeling with AI insights, expense management, and interactive charts.
          </p>
        </CardHeader>
      </Card>

      {/* Main Content Grid */}
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        {/* Left Column - Inputs and Configuration */}
        <div className="lg:col-span-2 space-y-6">
          {/* Enhanced Inputs & Expenses */}
          <Tabs defaultValue="inputs" className="w-full" data-testid="inputs-tabs">
            <TabsList className="grid w-full grid-cols-4">
              <TabsTrigger value="inputs" className="flex items-center gap-2">
                <Settings className="w-4 h-4" />
                Inputs
              </TabsTrigger>
              <TabsTrigger value="expenses" className="flex items-center gap-2">
                <Calculator className="w-4 h-4" />
                Expenses
              </TabsTrigger>
              <TabsTrigger value="import" className="flex items-center gap-2">
                <Upload className="w-4 h-4" />
                Import
              </TabsTrigger>
              <TabsTrigger value="charts" className="flex items-center gap-2">
                <PieChart className="w-4 h-4" />
                Charts
              </TabsTrigger>
            </TabsList>
            
            <TabsContent value="inputs" className="mt-4">
              <FinanceInputs />
            </TabsContent>
            
            <TabsContent value="expenses" className="mt-4">
              <ExpensesTable />
            </TabsContent>
            
            <TabsContent value="import" className="mt-4">
              <CsvImport />
            </TabsContent>
            
            <TabsContent value="charts" className="mt-4">
              <CustomChartBuilder targetSectionId={financialSectionId} />
            </TabsContent>
          </Tabs>

          {/* AI Insights */}
          <FinanceInsightsAI />
        </div>

        {/* Right Column - Summary */}
        <div className="space-y-6">
          <FinanceSidebar />
        </div>
      </div>

      {/* Charts Section */}
      <Card data-testid="charts-section">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <BarChart3 className="w-5 h-5" />
            Financial Visualizations
          </CardTitle>
        </CardHeader>
        <CardContent>
          <FinanceCharts />
        </CardContent>
      </Card>

      {/* Extended Analysis Tabs */}
      <Tabs defaultValue="overview" className="w-full" data-testid="analysis-tabs">
        <TabsList className="grid w-full grid-cols-2">
          <TabsTrigger value="overview" className="flex items-center gap-2">
            <BarChart3 className="w-4 h-4" />
            Performance Overview
          </TabsTrigger>
          <TabsTrigger value="insights" className="flex items-center gap-2">
            <Brain className="w-4 h-4" />
            AI Analysis
          </TabsTrigger>
        </TabsList>
        
        <TabsContent value="overview" className="mt-4">
          {/* Detailed Performance Analysis */}
          <PerformanceOverview />
        </TabsContent>
        
        <TabsContent value="insights" className="mt-4">
          {/* Enhanced AI Insights with more detail */}
          <AdvancedInsightsPanel />
        </TabsContent>
      </Tabs>
    </div>
  );
}

// Performance Overview Component
function PerformanceOverview() {
  const { inputs, outputs } = useFinanceStoreV2();

  if (!outputs) {
    return (
      <Card>
        <CardContent className="p-6 text-center">
          <p className="text-muted-foreground">Configure your financial inputs to see performance analysis.</p>
        </CardContent>
      </Card>
    );
  }

  const { rows, totalRevenue, totalProfit, breakevenMonthIndex, cashFlowAnalysis } = outputs;

  // Calculate key metrics
  const avgMonthlyRevenue = totalRevenue / 12;
  const avgMonthlyProfit = totalProfit / 12;
  const profitMargin = totalRevenue > 0 ? (totalProfit / totalRevenue) * 100 : 0;
  const revenueGrowth = rows.length > 1 ? ((rows[11].revenue - rows[0].revenue) / (rows[0].revenue || 1)) * 100 : 0;

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: inputs.currency,
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(amount);
  };

  return (
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4" data-testid="performance-overview">
      {/* Key Metrics Cards */}
      <Card>
        <CardContent className="p-4">
          <div className="text-sm text-muted-foreground">Avg Monthly Revenue</div>
          <div className="text-xl font-bold text-blue-600">{formatCurrency(avgMonthlyRevenue)}</div>
        </CardContent>
      </Card>

      <Card>
        <CardContent className="p-4">
          <div className="text-sm text-muted-foreground">Avg Monthly Profit</div>
          <div className={`text-xl font-bold ${avgMonthlyProfit >= 0 ? 'text-green-600' : 'text-red-600'}`}>
            {formatCurrency(avgMonthlyProfit)}
          </div>
        </CardContent>
      </Card>

      <Card>
        <CardContent className="p-4">
          <div className="text-sm text-muted-foreground">Profit Margin</div>
          <div className={`text-xl font-bold ${profitMargin >= 0 ? 'text-green-600' : 'text-red-600'}`}>
            {profitMargin.toFixed(1)}%
          </div>
        </CardContent>
      </Card>

      <Card>
        <CardContent className="p-4">
          <div className="text-sm text-muted-foreground">Revenue Growth</div>
          <div className={`text-xl font-bold ${revenueGrowth >= 0 ? 'text-green-600' : 'text-red-600'}`}>
            {revenueGrowth.toFixed(1)}%
          </div>
        </CardContent>
      </Card>

      {/* Cash Flow Health */}
      <Card className="md:col-span-2">
        <CardHeader>
          <CardTitle className="text-lg">Cash Flow Health</CardTitle>
        </CardHeader>
        <CardContent className="space-y-3">
          <div className="flex justify-between">
            <span className="text-muted-foreground">Starting Cash:</span>
            <span className="font-medium">{formatCurrency(cashFlowAnalysis.startingCash)}</span>
          </div>
          <div className="flex justify-between">
            <span className="text-muted-foreground">Ending Cash:</span>
            <span className={`font-medium ${cashFlowAnalysis.endingCash >= 0 ? 'text-green-600' : 'text-red-600'}`}>
              {formatCurrency(cashFlowAnalysis.endingCash)}
            </span>
          </div>
          <div className="flex justify-between">
            <span className="text-muted-foreground">Lowest Point:</span>
            <span className={`font-medium ${cashFlowAnalysis.lowestCash >= 0 ? 'text-green-600' : 'text-orange-600'}`}>
              {formatCurrency(cashFlowAnalysis.lowestCash)} (Month {cashFlowAnalysis.lowestCashMonth + 1})
            </span>
          </div>
        </CardContent>
      </Card>

      {/* Break-Even Analysis */}
      <Card className="md:col-span-2">
        <CardHeader>
          <CardTitle className="text-lg">Break-Even Analysis</CardTitle>
        </CardHeader>
        <CardContent className="space-y-3">
          {breakevenMonthIndex !== null ? (
            <>
              <div className="flex justify-between">
                <span className="text-muted-foreground">Break-Even Month:</span>
                <span className="font-medium text-green-600">Month {breakevenMonthIndex + 1}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-muted-foreground">Time to Profitability:</span>
                <span className="font-medium">{breakevenMonthIndex + 1} months</span>
              </div>
              <div className="flex justify-between">
                <span className="text-muted-foreground">Revenue at Break-Even:</span>
                <span className="font-medium">{formatCurrency(rows[breakevenMonthIndex].revenue)}</span>
              </div>
            </>
          ) : (
            <div className="text-center py-4">
              <div className="text-orange-600 font-medium">Break-even not reached</div>
              <div className="text-sm text-muted-foreground">Within 12-month projection period</div>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}

// Advanced Insights Panel Component  
function AdvancedInsightsPanel() {
  const { inputs, outputs } = useFinanceStoreV2();

  if (!outputs) {
    return (
      <Card>
        <CardContent className="p-6 text-center">
          <Brain className="w-12 h-12 mx-auto mb-3 opacity-50" />
          <p className="text-muted-foreground">Complete your financial model to unlock advanced AI analysis.</p>
        </CardContent>
      </Card>
    );
  }

  return (
    <div className="space-y-4" data-testid="advanced-insights">
      <FinanceInsightsAI />
      
      {/* Additional Analysis Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <Card>
          <CardHeader>
            <CardTitle className="text-lg">Unit Economics</CardTitle>
          </CardHeader>
          <CardContent className="space-y-2">
            <div className="flex justify-between">
              <span className="text-muted-foreground">Price per Unit:</span>
              <span className="font-medium">${inputs.pricePerUnit}</span>
            </div>
            <div className="flex justify-between">
              <span className="text-muted-foreground">COGS %:</span>
              <span className="font-medium">{inputs.cogsPercent}%</span>
            </div>
            <div className="flex justify-between">
              <span className="text-muted-foreground">Gross Margin:</span>
              <span className="font-medium">${(inputs.pricePerUnit * (100 - inputs.cogsPercent) / 100).toFixed(2)}</span>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="text-lg">Expense Breakdown</CardTitle>
          </CardHeader>
          <CardContent className="space-y-2">
            {inputs.expenses.slice(0, 3).map((expense, index) => (
              <div key={index} className="flex justify-between">
                <span className="text-muted-foreground">{expense.name}:</span>
                <span className="font-medium">${expense.amount}/mo</span>
              </div>
            ))}
            {inputs.expenses.length > 3 && (
              <div className="flex justify-between text-sm">
                <span className="text-muted-foreground">+ {inputs.expenses.length - 3} more</span>
                <span className="font-medium">${inputs.fixedMonthlyExpenses}/mo total</span>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </div>
  );
}