import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { useFinanceStoreV2 } from "@/stores/useFinanceStore.v2";
import { TrendingUp, Calendar, Clock, DollarSign } from "lucide-react";

function formatCurrency(amount: number, currency: string): string {
  return new Intl.NumberFormat('en-US', {
    style: 'currency',
    currency: currency,
    minimumFractionDigits: 0,
    maximumFractionDigits: 0,
  }).format(amount);
}

export function FinanceSidebar() {
  const { inputs, outputs } = useFinanceStoreV2();

  if (!outputs) {
    return (
      <Card data-testid="finance-sidebar-loading">
        <CardContent className="p-6 text-center">
          <p className="text-muted-foreground">Loading financial data...</p>
        </CardContent>
      </Card>
    );
  }

  const { rows, totalRevenue, totalProfit, breakevenMonthIndex, runwayMonths } = outputs;

  return (
    <div className="space-y-4" data-testid="finance-sidebar">
      {/* Summary Cards */}
      <div className="grid grid-cols-2 gap-3">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2 mb-2">
              <TrendingUp className="w-4 h-4 text-green-600" />
              <span className="text-sm font-medium">Total Revenue</span>
            </div>
            <div className="text-lg font-bold" data-testid="total-revenue">
              {formatCurrency(totalRevenue, inputs.currency)}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2 mb-2">
              <DollarSign className="w-4 h-4 text-blue-600" />
              <span className="text-sm font-medium">Total Profit</span>
            </div>
            <div className={`text-lg font-bold ${totalProfit >= 0 ? 'text-green-600' : 'text-red-600'}`} data-testid="total-profit">
              {formatCurrency(totalProfit, inputs.currency)}
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Breakeven and Runway */}
      <Card>
        <CardContent className="p-4 space-y-3">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-2">
              <Calendar className="w-4 h-4 text-orange-600" />
              <span className="text-sm font-medium">Breakeven</span>
            </div>
            <Badge variant={breakevenMonthIndex !== null ? "default" : "destructive"} data-testid="breakeven-badge">
              {breakevenMonthIndex !== null ? `Month ${breakevenMonthIndex + 1}` : "Not reached"}
            </Badge>
          </div>

          {runwayMonths !== null && (
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <Clock className="w-4 h-4 text-purple-600" />
                <span className="text-sm font-medium">Runway</span>
              </div>
              <Badge variant="outline" data-testid="runway-badge">
                {runwayMonths > 60 ? "60+ months" : `${runwayMonths} months`}
              </Badge>
            </div>
          )}
        </CardContent>
      </Card>

      {/* P&L Table */}
      <Card>
        <CardHeader>
          <CardTitle className="text-lg">12-Month P&L</CardTitle>
        </CardHeader>
        <CardContent className="p-0">
          <div className="overflow-x-auto">
            <table className="w-full text-xs" data-testid="pnl-table">
              <thead>
                <tr className="border-b bg-muted/50">
                  <th className="text-left p-2 font-medium">Month</th>
                  <th className="text-right p-2 font-medium">Revenue</th>
                  <th className="text-right p-2 font-medium">COGS</th>
                  <th className="text-right p-2 font-medium">Gross</th>
                  <th className="text-right p-2 font-medium">OpEx</th>
                  <th className="text-right p-2 font-medium">Profit</th>
                  <th className="text-right p-2 font-medium">Cumulative</th>
                </tr>
              </thead>
              <tbody>
                {rows.map((row, index) => (
                  <tr key={index} className="border-b hover:bg-muted/30" data-testid={`pnl-row-${index}`}>
                    <td className="p-2 font-medium">{row.month}</td>
                    <td className="text-right p-2">{formatCurrency(row.revenue, inputs.currency)}</td>
                    <td className="text-right p-2">{formatCurrency(row.cogs, inputs.currency)}</td>
                    <td className="text-right p-2">{formatCurrency(row.grossProfit, inputs.currency)}</td>
                    <td className="text-right p-2">{formatCurrency(row.opex, inputs.currency)}</td>
                    <td className={`text-right p-2 ${row.profit >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                      {formatCurrency(row.profit, inputs.currency)}
                    </td>
                    <td className={`text-right p-2 font-medium ${row.cumulativeProfit >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                      {formatCurrency(row.cumulativeProfit, inputs.currency)}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}