import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { useFinanceStoreV2 } from "@/stores/useFinanceStore.v2";
import { Brain, AlertCircle, TrendingUp, Target, Lightbulb, Loader2 } from "lucide-react";
import { useState } from "react";

interface FinanceInsight {
  title: string;
  content: string;
  type: "positive" | "warning" | "neutral";
  icon: "breakeven" | "driver" | "margin";
}

export function FinanceInsightsAI() {
  const { inputs, outputs } = useFinanceStoreV2();
  const [insights, setInsights] = useState<FinanceInsight[]>([]);
  const [isAnalyzing, setIsAnalyzing] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const generateInsights = async () => {
    if (!outputs) {
      setError("No financial data available for analysis");
      return;
    }

    setIsAnalyzing(true);
    setError(null);

    try {
      // Prepare structured financial data for AI analysis
      const financialData = {
        summary: {
          totalRevenue: outputs.totalRevenue,
          totalProfit: outputs.totalProfit,
          breakevenMonth: outputs.breakevenMonthIndex,
          currency: inputs.currency,
          pricePerUnit: inputs.pricePerUnit,
          cogsPercent: inputs.cogsPercent,
          fixedExpenses: inputs.fixedMonthlyExpenses,
        },
        monthlyData: outputs.rows.map(row => ({
          month: row.month,
          revenue: row.revenue,
          profit: row.profit,
          endingCash: row.endingCash,
        })),
        expenses: inputs.expenses,
        cashFlow: outputs.cashFlowAnalysis,
      };

      // Create AI prompt for financial analysis
      const prompt = `
Analyze this financial projection and provide exactly 3 actionable insights:

Financial Summary:
- Total 12-month revenue: ${formatCurrency(outputs.totalRevenue, inputs.currency)}
- Total 12-month profit: ${formatCurrency(outputs.totalProfit, inputs.currency)}
- Break-even: ${outputs.breakevenMonthIndex !== null ? `Month ${outputs.breakevenMonthIndex + 1}` : 'Not reached'}
- Price per unit: ${formatCurrency(inputs.pricePerUnit, inputs.currency)}
- COGS: ${inputs.cogsPercent}%
- Fixed monthly expenses: ${formatCurrency(inputs.fixedMonthlyExpenses, inputs.currency)}

Monthly Performance:
${outputs.rows.map(row => `${row.month}: Revenue ${formatCurrency(row.revenue, inputs.currency)}, Profit ${formatCurrency(row.profit, inputs.currency)}`).join('\n')}

Expense Breakdown:
${inputs.expenses.map(exp => `${exp.name}: ${formatCurrency(exp.amount, inputs.currency)}/month`).join('\n')}

Cash Flow:
- Starting cash: ${formatCurrency(outputs.cashFlowAnalysis.startingCash, inputs.currency)}
- Lowest cash: ${formatCurrency(outputs.cashFlowAnalysis.lowestCash, inputs.currency)} (Month ${outputs.cashFlowAnalysis.lowestCashMonth + 1})
- Ending cash: ${formatCurrency(outputs.cashFlowAnalysis.endingCash, inputs.currency)}

Please provide exactly 3 insights in this format:

INSIGHT 1 - BREAKEVEN ANALYSIS:
[Analysis of break-even timing and what's driving it]

INSIGHT 2 - PROFIT DRIVERS:
[Identify the biggest factors affecting profitability - revenue growth, COGS, expenses]

INSIGHT 3 - MARGIN IMPROVEMENT:
[Specific actionable recommendation to improve margins - could be pricing, cost reduction, efficiency]

Keep each insight concise (2-3 sentences) and actionable.
      `.trim();

      // Use the existing AI section endpoint for financial analysis
      const response = await fetch('/api/ai/section', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        credentials: 'include',
        body: JSON.stringify({
          action: 'generate',
          tone: 'Professional',
          sectionKind: 'financials',
          sectionTitle: 'Financial Analysis',
          existingContent: prompt,
          context: {
            businessName: 'Financial Analysis',
            financialData
          }
        }),
      });

      if (!response.ok) {
        // Fallback to generated insights if AI endpoint isn't available
        const generatedInsights = generateFallbackInsights();
        setInsights(generatedInsights);
        return;
      }

      const data = await response.json();
      const parsedInsights = parseAIInsights(data.content || data.insights || '');
      setInsights(parsedInsights);

    } catch (error) {
      console.error('AI insights error:', error);
      // Generate fallback insights based on the data
      const fallbackInsights = generateFallbackInsights();
      setInsights(fallbackInsights);
    } finally {
      setIsAnalyzing(false);
    }
  };

  const generateFallbackInsights = (): FinanceInsight[] => {
    if (!outputs) return [];

    const insights: FinanceInsight[] = [];

    // Breakeven Analysis
    if (outputs.breakevenMonthIndex !== null) {
      insights.push({
        title: "Break-Even Analysis",
        content: `You'll reach break-even in month ${outputs.breakevenMonthIndex + 1}. This is ${outputs.breakevenMonthIndex < 6 ? 'relatively quick' : 'on the longer side'} - focus on ${outputs.breakevenMonthIndex < 6 ? 'maintaining momentum' : 'accelerating unit sales or reducing costs'} to improve this timeline.`,
        type: outputs.breakevenMonthIndex < 6 ? "positive" : "warning",
        icon: "breakeven",
      });
    } else {
      insights.push({
        title: "Break-Even Analysis",
        content: `Break-even isn't reached within 12 months. Your fixed costs (${formatCurrency(inputs.fixedMonthlyExpenses, inputs.currency)}/month) may be too high relative to unit economics. Consider reducing expenses or increasing pricing/volume.`,
        type: "warning",
        icon: "breakeven",
      });
    }

    // Profit Drivers
    const avgRevenue = outputs.totalRevenue / 12;
    const marginPercent = ((avgRevenue - (avgRevenue * inputs.cogsPercent / 100) - inputs.fixedMonthlyExpenses) / avgRevenue) * 100;
    
    insights.push({
      title: "Profit Drivers",
      content: `Your current gross margin after COGS is ${(100 - inputs.cogsPercent).toFixed(1)}%. With fixed costs, your net margin is ${marginPercent.toFixed(1)}%. ${marginPercent > 20 ? 'Strong margins' : marginPercent > 10 ? 'Decent margins' : 'Thin margins'} - ${marginPercent <= 10 ? 'focus on efficiency or pricing power' : 'you have good unit economics'}.`,
      type: marginPercent > 15 ? "positive" : marginPercent > 5 ? "neutral" : "warning",
      icon: "driver",
    });

    // Margin Improvement
    const largestExpense = inputs.expenses.reduce((max, exp) => exp.amount > max.amount ? exp : max, inputs.expenses[0]);
    
    insights.push({
      title: "Margin Improvement",
      content: `Your largest expense is ${largestExpense?.name} at ${formatCurrency(largestExpense?.amount || 0, inputs.currency)}/month. ${inputs.cogsPercent > 40 ? 'Also consider optimizing your COGS (currently ' + inputs.cogsPercent + '%)' : 'Review if you can optimize this cost'} or test a ${((inputs.pricePerUnit * 1.1) - inputs.pricePerUnit).toFixed(0)}% price increase.`,
      type: "neutral",
      icon: "margin",
    });

    return insights;
  };

  const parseAIInsights = (content: string): FinanceInsight[] => {
    const insights: FinanceInsight[] = [];
    
    // Parse the AI response for the three insights
    const sections = content.split(/INSIGHT \d+/);
    
    if (sections.length >= 4) {
      const titles = ["Break-Even Analysis", "Profit Drivers", "Margin Improvement"];
      const icons: ("breakeven" | "driver" | "margin")[] = ["breakeven", "driver", "margin"];
      
      for (let i = 1; i <= 3; i++) {
        const section = sections[i]?.replace(/^[\s\-:]+/, '').trim();
        if (section) {
          insights.push({
            title: titles[i - 1],
            content: section,
            type: "neutral",
            icon: icons[i - 1],
          });
        }
      }
    }

    // If parsing fails, return fallback
    if (insights.length === 0) {
      return generateFallbackInsights();
    }

    return insights;
  };

  const getIcon = (iconType: string) => {
    switch (iconType) {
      case "breakeven": return <Target className="w-4 h-4" />;
      case "driver": return <TrendingUp className="w-4 h-4" />;
      case "margin": return <Lightbulb className="w-4 h-4" />;
      default: return <Brain className="w-4 h-4" />;
    }
  };

  const getInsightColor = (type: string) => {
    switch (type) {
      case "positive": return "text-green-700 bg-green-50 border-green-200";
      case "warning": return "text-orange-700 bg-orange-50 border-orange-200";
      case "neutral": return "text-blue-700 bg-blue-50 border-blue-200";
      default: return "text-gray-700 bg-gray-50 border-gray-200";
    }
  };

  function formatCurrency(amount: number, currency: string): string {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: currency,
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(amount);
  }

  return (
    <Card data-testid="finance-insights-ai">
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Brain className="w-5 h-5" />
          AI Financial Insights
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-4">
        {/* Analyze Button */}
        <div className="flex items-center gap-3">
          <Button
            onClick={generateInsights}
            disabled={!outputs || isAnalyzing}
            className="flex items-center gap-2"
            data-testid="button-analyze-finances"
          >
            {isAnalyzing ? (
              <Loader2 className="w-4 h-4 animate-spin" />
            ) : (
              <Brain className="w-4 h-4" />
            )}
            {isAnalyzing ? "Analyzing..." : "Analyze Finances"}
          </Button>
          {!outputs && (
            <span className="text-sm text-muted-foreground">
              Complete your financial inputs first
            </span>
          )}
        </div>

        {/* Error State */}
        {error && (
          <div className="flex items-center gap-2 text-sm p-3 rounded-lg bg-red-50 text-red-700 border border-red-200" data-testid="insights-error">
            <AlertCircle className="w-4 h-4" />
            <span>{error}</span>
          </div>
        )}

        {/* Insights */}
        {insights.length > 0 && (
          <div className="space-y-3" data-testid="insights-list">
            {insights.map((insight, index) => (
              <div
                key={index}
                className={`p-4 rounded-lg border ${getInsightColor(insight.type)}`}
                data-testid={`insight-${index}`}
              >
                <div className="flex items-start gap-3">
                  <div className="mt-0.5">
                    {getIcon(insight.icon)}
                  </div>
                  <div className="flex-1">
                    <h4 className="font-semibold text-sm mb-1">{insight.title}</h4>
                    <p className="text-sm leading-relaxed">{insight.content}</p>
                  </div>
                </div>
              </div>
            ))}
          </div>
        )}

        {/* Empty State */}
        {insights.length === 0 && !isAnalyzing && !error && (
          <div className="text-center py-8 text-muted-foreground">
            <Brain className="w-12 h-12 mx-auto mb-3 opacity-50" />
            <p className="text-sm">Click "Analyze Finances" to get AI-powered insights about your financial projections.</p>
          </div>
        )}
      </CardContent>
    </Card>
  );
}