import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Button } from "@/components/ui/button";
import { useFinanceStoreV2 } from "@/stores/useFinanceStore.v2";
import { Calculator, RotateCcw } from "lucide-react";

export function FinanceInputs() {
  const { inputs, setInputs, recompute, reset } = useFinanceStoreV2();

  const handleInputChange = (field: keyof typeof inputs, value: any) => {
    setInputs({ [field]: value });
    // Auto-recompute after a brief delay
    setTimeout(recompute, 100);
  };

  const handleUnitsChange = (index: number, value: string) => {
    const newUnits = [...inputs.unitsPerMonth];
    newUnits[index] = Number(value) || 0;
    handleInputChange('unitsPerMonth', newUnits);
  };

  return (
    <Card data-testid="finance-inputs">
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Calculator className="w-5 h-5" />
          Financial Inputs
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-4">
        {/* Currency and Start Month */}
        <div className="grid grid-cols-2 gap-3">
          <div className="space-y-2">
            <Label htmlFor="currency">Currency</Label>
            <Select value={inputs.currency} onValueChange={(value) => handleInputChange('currency', value)}>
              <SelectTrigger data-testid="select-currency">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="USD">USD ($)</SelectItem>
                <SelectItem value="EUR">EUR (€)</SelectItem>
                <SelectItem value="GBP">GBP (£)</SelectItem>
                <SelectItem value="CAD">CAD ($)</SelectItem>
                <SelectItem value="AUD">AUD ($)</SelectItem>
              </SelectContent>
            </Select>
          </div>
          <div className="space-y-2">
            <Label htmlFor="startMonth">Start Month</Label>
            <Input
              id="startMonth"
              type="month"
              value={inputs.startMonth}
              onChange={(e) => handleInputChange('startMonth', e.target.value)}
              data-testid="input-start-month"
            />
          </div>
        </div>

        {/* Price and COGS */}
        <div className="grid grid-cols-2 gap-3">
          <div className="space-y-2">
            <Label htmlFor="pricePerUnit">Price per Unit</Label>
            <Input
              id="pricePerUnit"
              type="number"
              value={inputs.pricePerUnit}
              onChange={(e) => handleInputChange('pricePerUnit', Number(e.target.value))}
              min="0"
              step="0.01"
              data-testid="input-price-per-unit"
            />
          </div>
          <div className="space-y-2">
            <Label htmlFor="cogsPercent">COGS %</Label>
            <Input
              id="cogsPercent"
              type="number"
              value={inputs.cogsPercent}
              onChange={(e) => handleInputChange('cogsPercent', Number(e.target.value))}
              min="0"
              max="100"
              step="0.1"
              data-testid="input-cogs-percent"
            />
          </div>
        </div>

        {/* Fixed and Variable Expenses */}
        <div className="grid grid-cols-2 gap-3">
          <div className="space-y-2">
            <Label htmlFor="fixedMonthlyExpenses">Fixed Monthly Expenses</Label>
            <Input
              id="fixedMonthlyExpenses"
              type="number"
              value={inputs.fixedMonthlyExpenses}
              onChange={(e) => handleInputChange('fixedMonthlyExpenses', Number(e.target.value))}
              min="0"
              step="0.01"
              data-testid="input-fixed-expenses"
            />
          </div>
          <div className="space-y-2">
            <Label htmlFor="variableExpensePerUnit">Variable Cost per Unit</Label>
            <Input
              id="variableExpensePerUnit"
              type="number"
              value={inputs.variableExpensePerUnit || 0}
              onChange={(e) => handleInputChange('variableExpensePerUnit', Number(e.target.value))}
              min="0"
              step="0.01"
              data-testid="input-variable-expense"
            />
          </div>
        </div>

        {/* Starting Cash */}
        <div className="space-y-2">
          <Label htmlFor="startingCash">Starting Cash (optional)</Label>
          <Input
            id="startingCash"
            type="number"
            value={inputs.startingCash || 0}
            onChange={(e) => handleInputChange('startingCash', Number(e.target.value))}
            min="0"
            step="0.01"
            data-testid="input-starting-cash"
          />
        </div>

        {/* Units per Month */}
        <div className="space-y-3">
          <Label>Units Sold per Month</Label>
          <div className="grid grid-cols-3 gap-2">
            {inputs.unitsPerMonth.map((units, index) => (
              <div key={index} className="space-y-1">
                <Label className="text-xs text-muted-foreground">
                  Month {index + 1}
                </Label>
                <Input
                  type="number"
                  value={units}
                  onChange={(e) => handleUnitsChange(index, e.target.value)}
                  min="0"
                  className="text-sm"
                  data-testid={`input-units-month-${index}`}
                />
              </div>
            ))}
          </div>
        </div>

        {/* Actions */}
        <div className="flex gap-2 pt-4">
          <Button
            onClick={recompute}
            className="flex-1"
            data-testid="button-recompute"
          >
            <Calculator className="w-4 h-4 mr-2" />
            Recompute
          </Button>
          <Button
            onClick={reset}
            variant="outline"
            data-testid="button-reset"
          >
            <RotateCcw className="w-4 h-4" />
          </Button>
        </div>
      </CardContent>
    </Card>
  );
}