import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { useFinanceStoreV2 } from "@/stores/useFinanceStore.v2";
import { usePlanStoreV2 } from "@/stores/usePlanStore.v2";
import { useBrandKitStore } from "@/stores/useBrandKitStore";
import { getBrandPalette } from "@/lib/brand";
import { 
  LineChart, 
  Line, 
  AreaChart, 
  Area, 
  BarChart, 
  Bar, 
  XAxis, 
  YAxis, 
  CartesianGrid, 
  Tooltip, 
  ResponsiveContainer,
  ReferenceLine
} from "recharts";
import { TrendingUp, Activity, Target } from "lucide-react";
import { useRef } from "react";
import { ChartSnapshotButton } from "@/components/ChartSnapshotButton";

function formatCurrency(amount: number, currency: string): string {
  return new Intl.NumberFormat('en-US', {
    style: 'currency',
    currency: currency,
    minimumFractionDigits: 0,
    maximumFractionDigits: 0,
  }).format(amount);
}

export function FinanceCharts() {
  const { inputs, outputs } = useFinanceStoreV2();
  const { sections } = usePlanStoreV2();
  const brand = useBrandKitStore((s) => s.getCurrentKit());
  const refRevenue = useRef<HTMLDivElement>(null);
  const refCashFlow = useRef<HTMLDivElement>(null);
  const refBreakeven = useRef<HTMLDivElement>(null);
  
  // Find the Financial Plan section for inserting snapshots
  const financialSection = sections.find(s => s.kind === "financials");
  const financialSectionId = financialSection?.id || "";

  // Get brand-aware color palette
  const palette = getBrandPalette(brand);

  if (!outputs) {
    return (
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4" data-testid="finance-charts-loading">
        <Card className="col-span-full">
          <CardContent className="p-6 text-center">
            <p className="text-muted-foreground">Calculating financial projections...</p>
          </CardContent>
        </Card>
      </div>
    );
  }

  const { rows, breakevenMonthIndex, cashFlowAnalysis } = outputs;

  // Chart data preparation
  const revenueVsProfitData = rows.map((row, index) => ({
    month: row.month.split(' ')[0], // Short month name
    revenue: row.revenue,
    profit: row.profit,
    monthIndex: index,
  }));

  const cashFlowData = rows.map((row, index) => ({
    month: row.month.split(' ')[0],
    endingCash: row.endingCash,
    profit: row.profit,
    monthIndex: index,
  }));

  const breakevenData = rows.map((row, index) => ({
    month: row.month.split(' ')[0],
    cumulativeProfit: row.cumulativeProfit,
    isBreakeven: index === breakevenMonthIndex,
    monthIndex: index,
  }));

  return (
    <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-3 gap-4" data-testid="finance-charts">
      {/* Revenue vs Profit Chart */}
      <Card className="lg:col-span-2" data-testid="revenue-profit-chart">
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle className="flex items-center gap-2">
              <TrendingUp className="w-5 h-5" />
              Revenue vs Profit
            </CardTitle>
            <ChartSnapshotButton 
              targetSectionId={financialSectionId} 
              getElement={() => refRevenue.current} 
              title="Revenue vs Profit" 
            />
          </div>
        </CardHeader>
        <CardContent>
          <div ref={refRevenue} style={{ fontFamily: brand?.fontBody || "Inter, ui-sans-serif" }}>
            <ResponsiveContainer width="100%" height={300}>
            <LineChart data={revenueVsProfitData}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis 
                dataKey="month" 
                tick={{ fontSize: 12 }}
              />
              <YAxis 
                tick={{ fontSize: 12 }}
                tickFormatter={(value) => formatCurrency(value, inputs.currency)}
              />
              <Tooltip 
                formatter={(value: number, name: string) => [
                  formatCurrency(value, inputs.currency), 
                  name === 'revenue' ? 'Revenue' : 'Profit'
                ]}
                labelFormatter={(label) => `Month: ${label}`}
              />
              <Line 
                type="monotone" 
                dataKey="revenue" 
                stroke={palette[0]} 
                strokeWidth={3}
                dot={{ fill: palette[0], strokeWidth: 2, r: 4 }}
                name="revenue"
              />
              <Line 
                type="monotone" 
                dataKey="profit" 
                stroke={palette[1]} 
                strokeWidth={3}
                dot={{ fill: palette[1], strokeWidth: 2, r: 4 }}
                name="profit"
              />
              <ReferenceLine y={0} stroke="#666" strokeDasharray="2 2" />
            </LineChart>
          </ResponsiveContainer>
          </div>
        </CardContent>
      </Card>

      {/* Cash Flow Chart */}
      <Card data-testid="cash-flow-chart">
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle className="flex items-center gap-2">
              <Activity className="w-5 h-5" />
              Cash Flow
            </CardTitle>
            <ChartSnapshotButton 
              targetSectionId={financialSectionId} 
              getElement={() => refCashFlow.current} 
              title="Cash Flow (Ending Cash)" 
            />
          </div>
        </CardHeader>
        <CardContent>
          <div ref={refCashFlow} style={{ fontFamily: brand?.fontBody || "Inter, ui-sans-serif" }}>
            <ResponsiveContainer width="100%" height={300}>
            <AreaChart data={cashFlowData}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis 
                dataKey="month" 
                tick={{ fontSize: 12 }}
              />
              <YAxis 
                tick={{ fontSize: 12 }}
                tickFormatter={(value) => formatCurrency(value, inputs.currency)}
              />
              <Tooltip 
                formatter={(value: number, name: string) => [
                  formatCurrency(value, inputs.currency), 
                  name === 'endingCash' ? 'Ending Cash' : 'Monthly Profit'
                ]}
                labelFormatter={(label) => `Month: ${label}`}
              />
              <Area 
                type="monotone" 
                dataKey="endingCash" 
                stroke={palette[2]} 
                fill={palette[2]} 
                fillOpacity={0.3}
                strokeWidth={2}
                name="endingCash"
              />
              <ReferenceLine y={0} stroke="#dc2626" strokeDasharray="2 2" />
            </AreaChart>
          </ResponsiveContainer>
          
          {/* Cash Flow Summary */}
          <div className="mt-3 p-3 bg-muted/30 rounded-lg">
            <div className="grid grid-cols-2 gap-2 text-xs">
              <div>
                <span className="text-muted-foreground">Starting:</span>
                <div className="font-medium">{formatCurrency(cashFlowAnalysis.startingCash, inputs.currency)}</div>
              </div>
              <div>
                <span className="text-muted-foreground">Ending:</span>
                <div className="font-medium">{formatCurrency(cashFlowAnalysis.endingCash, inputs.currency)}</div>
              </div>
              <div>
                <span className="text-muted-foreground">Lowest:</span>
                <div className="font-medium text-orange-600">{formatCurrency(cashFlowAnalysis.lowestCash, inputs.currency)}</div>
              </div>
              <div>
                <span className="text-muted-foreground">Risk Month:</span>
                <div className="font-medium">{rows[cashFlowAnalysis.lowestCashMonth]?.month.split(' ')[0]}</div>
              </div>
            </div>
          </div>
          </div>
        </CardContent>
      </Card>

      {/* Break-Even Analysis */}
      <Card className="lg:col-span-2 xl:col-span-1" data-testid="breakeven-chart">
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle className="flex items-center gap-2">
              <Target className="w-5 h-5" />
              Break-Even Analysis
            </CardTitle>
            <ChartSnapshotButton 
              targetSectionId={financialSectionId} 
              getElement={() => refBreakeven.current} 
              title="Break-Even (Profit by Month)" 
            />
          </div>
        </CardHeader>
        <CardContent>
          <div ref={refBreakeven} style={{ fontFamily: brand?.fontBody || "Inter, ui-sans-serif" }}>
            <ResponsiveContainer width="100%" height={250}>
            <BarChart data={breakevenData}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis 
                dataKey="month" 
                tick={{ fontSize: 12 }}
              />
              <YAxis 
                tick={{ fontSize: 12 }}
                tickFormatter={(value) => formatCurrency(value, inputs.currency)}
              />
              <Tooltip 
                formatter={(value: number) => [
                  formatCurrency(value, inputs.currency), 
                  'Cumulative Profit'
                ]}
                labelFormatter={(label) => `Month: ${label}`}
              />
              <Bar 
                dataKey="cumulativeProfit" 
                fill={palette[2]}
                radius={[2, 2, 0, 0]}
              />
              <ReferenceLine y={0} stroke="#666" strokeDasharray="2 2" />
            </BarChart>
          </ResponsiveContainer>

          {/* Break-Even Summary */}
          <div className="mt-3 p-3 bg-muted/30 rounded-lg">
            {breakevenMonthIndex !== null ? (
              <div className="text-center">
                <div className="text-sm text-muted-foreground">Break-even reached in</div>
                <div className="text-lg font-bold text-green-600">
                  Month {breakevenMonthIndex + 1}
                </div>
                <div className="text-xs text-muted-foreground">
                  {rows[breakevenMonthIndex]?.month}
                </div>
              </div>
            ) : (
              <div className="text-center">
                <div className="text-sm text-muted-foreground">Break-even</div>
                <div className="text-lg font-bold text-orange-600">
                  Not Reached
                </div>
                <div className="text-xs text-muted-foreground">
                  Within 12 months
                </div>
              </div>
            )}
          </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}