import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { useFinanceStoreV2 } from "@/stores/useFinanceStore.v2";
import { Plus, Trash2, DollarSign } from "lucide-react";
import { useState } from "react";

function formatCurrency(amount: number, currency: string): string {
  return new Intl.NumberFormat('en-US', {
    style: 'currency',
    currency: currency,
    minimumFractionDigits: 0,
    maximumFractionDigits: 0,
  }).format(amount);
}

export function ExpensesTable() {
  const { inputs, addExpense, updateExpense, deleteExpense, recompute } = useFinanceStoreV2();
  const [editingId, setEditingId] = useState<string | null>(null);

  const totalExpenses = inputs.expenses.reduce((sum, exp) => sum + (exp.amount || 0), 0);

  const handleNameChange = (id: string, name: string) => {
    updateExpense(id, { name });
    setTimeout(recompute, 100);
  };

  const handleAmountChange = (id: string, amount: number) => {
    updateExpense(id, { amount });
    setTimeout(recompute, 100);
  };

  const handleDelete = (id: string) => {
    deleteExpense(id);
    setTimeout(recompute, 100);
  };

  const handleAdd = () => {
    addExpense("New Expense", 0);
    setTimeout(recompute, 100);
  };

  const handleAmountFocus = (id: string) => {
    setEditingId(id);
  };

  const handleAmountBlur = () => {
    setEditingId(null);
  };

  return (
    <Card data-testid="expenses-table">
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <DollarSign className="w-5 h-5" />
          Monthly Expenses
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-4">
        {/* Table Header */}
        <div className="grid grid-cols-12 gap-3 text-sm font-medium text-muted-foreground border-b pb-2">
          <div className="col-span-6">Expense Name</div>
          <div className="col-span-4 text-right">Amount/mo</div>
          <div className="col-span-2 text-center">Actions</div>
        </div>

        {/* Expense Rows */}
        <div className="space-y-2" data-testid="expense-rows">
          {inputs.expenses.map((expense) => (
            <div key={expense.id} className="grid grid-cols-12 gap-3 items-center py-2 hover:bg-muted/30 rounded-lg px-2" data-testid={`expense-row-${expense.id}`}>
              {/* Name Input */}
              <div className="col-span-6">
                <Input
                  value={expense.name}
                  onChange={(e) => handleNameChange(expense.id, e.target.value)}
                  className="border-none bg-transparent p-1 h-8 text-sm"
                  placeholder="Expense name"
                  data-testid={`input-expense-name-${expense.id}`}
                />
              </div>

              {/* Amount Input */}
              <div className="col-span-4 text-right">
                <Input
                  type="number"
                  value={editingId === expense.id ? expense.amount : expense.amount || ''}
                  onChange={(e) => handleAmountChange(expense.id, Number(e.target.value) || 0)}
                  onFocus={() => handleAmountFocus(expense.id)}
                  onBlur={handleAmountBlur}
                  className="border-none bg-transparent p-1 h-8 text-sm text-right"
                  placeholder="0"
                  min="0"
                  step="0.01"
                  data-testid={`input-expense-amount-${expense.id}`}
                />
              </div>

              {/* Delete Button */}
              <div className="col-span-2 text-center">
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={() => handleDelete(expense.id)}
                  className="h-8 w-8 p-0 text-destructive hover:text-destructive"
                  data-testid={`button-delete-expense-${expense.id}`}
                >
                  <Trash2 className="w-4 h-4" />
                </Button>
              </div>
            </div>
          ))}
        </div>

        {/* Add New Expense Button */}
        <div className="pt-2">
          <Button
            variant="outline"
            onClick={handleAdd}
            className="w-full flex items-center gap-2"
            data-testid="button-add-expense"
          >
            <Plus className="w-4 h-4" />
            Add Expense
          </Button>
        </div>

        {/* Total */}
        <div className="border-t pt-4">
          <div className="flex justify-between items-center font-semibold">
            <span>Fixed Monthly Total:</span>
            <span className="text-lg" data-testid="total-expenses">
              {formatCurrency(totalExpenses, inputs.currency)}
            </span>
          </div>
          <p className="text-xs text-muted-foreground mt-1">
            This total is automatically used in your P&L calculations
          </p>
        </div>
      </CardContent>
    </Card>
  );
}