import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import { useFinanceStoreV2 } from "@/stores/useFinanceStore.v2";
import { Upload, FileText, AlertCircle, CheckCircle } from "lucide-react";
import { useState } from "react";
import Papa from "papaparse";
import { nanoid } from "nanoid";
import type { ExpenseRow } from "@shared/finance-types";

interface ImportResult {
  success: boolean;
  message: string;
  rowsProcessed?: number;
}

export function CsvImport() {
  const { inputs, setInputs, setExpenses, recompute } = useFinanceStoreV2();
  const [unitsResult, setUnitsResult] = useState<ImportResult | null>(null);
  const [expensesResult, setExpensesResult] = useState<ImportResult | null>(null);
  const [isProcessing, setIsProcessing] = useState(false);

  const handleUnitsImport = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    setIsProcessing(true);
    setUnitsResult(null);

    Papa.parse(file, {
      header: true,
      skipEmptyLines: true,
      complete: (results) => {
        try {
          if (results.errors.length > 0) {
            throw new Error(`CSV parsing error: ${results.errors[0].message}`);
          }

          const units = new Array(12).fill(0);
          let rowsProcessed = 0;

          results.data.forEach((row: any, index) => {
            const month = Number(row.month) || (index + 1);
            const unitsValue = Number(row.units) || 0;
            
            if (month >= 1 && month <= 12) {
              units[month - 1] = unitsValue;
              rowsProcessed++;
            }
          });

          if (rowsProcessed === 0) {
            throw new Error("No valid data found. Expected columns: month, units");
          }

          setInputs({ unitsPerMonth: units });
          setTimeout(recompute, 100);

          setUnitsResult({
            success: true,
            message: `Successfully imported ${rowsProcessed} months of unit data`,
            rowsProcessed,
          });
        } catch (error) {
          setUnitsResult({
            success: false,
            message: error instanceof Error ? error.message : "Import failed",
          });
        } finally {
          setIsProcessing(false);
          // Clear the input
          event.target.value = '';
        }
      },
      error: (error) => {
        setUnitsResult({
          success: false,
          message: `File parsing error: ${error.message}`,
        });
        setIsProcessing(false);
        event.target.value = '';
      }
    });
  };

  const handleExpensesImport = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    setIsProcessing(true);
    setExpensesResult(null);

    Papa.parse(file, {
      header: true,
      skipEmptyLines: true,
      complete: (results) => {
        try {
          if (results.errors.length > 0) {
            throw new Error(`CSV parsing error: ${results.errors[0].message}`);
          }

          const expenses: ExpenseRow[] = [];
          let rowsProcessed = 0;

          results.data.forEach((row: any) => {
            const name = String(row.name || row.expense || '').trim();
            const amount = Number(row.amount || row.cost || 0);
            
            if (name && amount > 0) {
              expenses.push({
                id: nanoid(),
                name,
                amount,
              });
              rowsProcessed++;
            }
          });

          if (rowsProcessed === 0) {
            throw new Error("No valid data found. Expected columns: name, amount (or expense, cost)");
          }

          setExpenses(expenses);
          setTimeout(recompute, 100);

          setExpensesResult({
            success: true,
            message: `Successfully imported ${rowsProcessed} expenses`,
            rowsProcessed,
          });
        } catch (error) {
          setExpensesResult({
            success: false,
            message: error instanceof Error ? error.message : "Import failed",
          });
        } finally {
          setIsProcessing(false);
          // Clear the input
          event.target.value = '';
        }
      },
      error: (error) => {
        setExpensesResult({
          success: false,
          message: `File parsing error: ${error.message}`,
        });
        setIsProcessing(false);
        event.target.value = '';
      }
    });
  };

  const clearResults = () => {
    setUnitsResult(null);
    setExpensesResult(null);
  };

  return (
    <Card data-testid="csv-import">
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Upload className="w-5 h-5" />
          Import Data
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-6">
        {/* Units Import Section */}
        <div className="space-y-3">
          <div className="space-y-2">
            <Label htmlFor="units-csv" className="text-sm font-medium">
              Import Monthly Units
            </Label>
            <div className="text-xs text-muted-foreground space-y-1">
              <p>Expected CSV format:</p>
              <code className="bg-muted px-2 py-1 rounded text-xs">month,units</code>
              <p>Example: 1,10 | 2,12 | 3,15 ...</p>
            </div>
          </div>
          
          <div className="flex items-center gap-3">
            <Input
              id="units-csv"
              type="file"
              accept=".csv"
              onChange={handleUnitsImport}
              disabled={isProcessing}
              className="flex-1"
              data-testid="input-units-csv"
            />
            <FileText className="w-4 h-4 text-muted-foreground" />
          </div>

          {unitsResult && (
            <div className={`flex items-center gap-2 text-sm p-2 rounded-lg ${
              unitsResult.success 
                ? 'bg-green-50 text-green-700 border border-green-200' 
                : 'bg-red-50 text-red-700 border border-red-200'
            }`} data-testid="units-import-result">
              {unitsResult.success ? (
                <CheckCircle className="w-4 h-4" />
              ) : (
                <AlertCircle className="w-4 h-4" />
              )}
              <span>{unitsResult.message}</span>
            </div>
          )}
        </div>

        {/* Expenses Import Section */}
        <div className="space-y-3">
          <div className="space-y-2">
            <Label htmlFor="expenses-csv" className="text-sm font-medium">
              Import Monthly Expenses
            </Label>
            <div className="text-xs text-muted-foreground space-y-1">
              <p>Expected CSV format:</p>
              <code className="bg-muted px-2 py-1 rounded text-xs">name,amount</code>
              <p>Example: Rent,1200 | Software,800 | Marketing,500</p>
            </div>
          </div>
          
          <div className="flex items-center gap-3">
            <Input
              id="expenses-csv"
              type="file"
              accept=".csv"
              onChange={handleExpensesImport}
              disabled={isProcessing}
              className="flex-1"
              data-testid="input-expenses-csv"
            />
            <FileText className="w-4 h-4 text-muted-foreground" />
          </div>

          {expensesResult && (
            <div className={`flex items-center gap-2 text-sm p-2 rounded-lg ${
              expensesResult.success 
                ? 'bg-green-50 text-green-700 border border-green-200' 
                : 'bg-red-50 text-red-700 border border-red-200'
            }`} data-testid="expenses-import-result">
              {expensesResult.success ? (
                <CheckCircle className="w-4 h-4" />
              ) : (
                <AlertCircle className="w-4 h-4" />
              )}
              <span>{expensesResult.message}</span>
            </div>
          )}
        </div>

        {/* Download Templates */}
        <div className="border-t pt-4">
          <div className="space-y-2">
            <Label className="text-sm font-medium">Sample CSV Templates</Label>
            <div className="flex gap-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => {
                  const csvContent = "month,units\n1,10\n2,12\n3,15\n4,18\n5,22\n6,26\n7,30\n8,35\n9,40\n10,46\n11,52\n12,60";
                  const blob = new Blob([csvContent], { type: 'text/csv' });
                  const url = URL.createObjectURL(blob);
                  const a = document.createElement('a');
                  a.href = url;
                  a.download = 'units-template.csv';
                  a.click();
                  URL.revokeObjectURL(url);
                }}
                data-testid="button-download-units-template"
              >
                Units Template
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => {
                  const csvContent = "name,amount\nRent & Utilities,1200\nSoftware & Tools,800\nMarketing,500\nMisc,500";
                  const blob = new Blob([csvContent], { type: 'text/csv' });
                  const url = URL.createObjectURL(blob);
                  const a = document.createElement('a');
                  a.href = url;
                  a.download = 'expenses-template.csv';
                  a.click();
                  URL.revokeObjectURL(url);
                }}
                data-testid="button-download-expenses-template"
              >
                Expenses Template
              </Button>
            </div>
          </div>
        </div>

        {/* Clear Results */}
        {(unitsResult || expensesResult) && (
          <div className="border-t pt-4">
            <Button
              variant="ghost"
              size="sm"
              onClick={clearResults}
              data-testid="button-clear-results"
            >
              Clear Results
            </Button>
          </div>
        )}

        {isProcessing && (
          <div className="text-center py-4">
            <div className="inline-flex items-center gap-2 text-sm text-muted-foreground">
              <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-primary"></div>
              Processing CSV...
            </div>
          </div>
        )}
      </CardContent>
    </Card>
  );
}