import { useState } from "react";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Checkbox } from "@/components/ui/checkbox";
import { Label } from "@/components/ui/label";
import { Separator } from "@/components/ui/separator";
import { ScrollArea } from "@/components/ui/scroll-area";
import { useToast } from "@/hooks/use-toast";
import { Loader2 } from "lucide-react";
import { normalizeSvg } from "@/lib/exporters/svg";
import { svgToPng } from "@/lib/exporters/png";
import { pngToJpeg } from "@/lib/exporters/jpeg";
import { svgToPdf } from "@/lib/exporters/pdf";
import { svgToEps } from "@/lib/exporters/eps";
import { pngToWebp } from "@/lib/exporters/webp";
import { createZip, type ZipFile } from "@/lib/zip";

interface ExportModalProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  svgMarkup: string;
  brandName: string;
}

interface ExportFormats {
  svg: boolean;
  png: boolean;
  jpeg: boolean;
  pdf: boolean;
  eps: boolean;
  webp: boolean;
}

interface PngSizes {
  256: boolean;
  512: boolean;
  1024: boolean;
}

const README_CONTENT = `Common Logo File Types

SVG → Vector (scales infinitely). Editable in Figma/Illustrator. Essential for web/app usage.
PNG → Raster, transparent background. For websites, presentations, docs. Included sizes: 256/512/1024 px.
JPEG → Raster, compressed (no transparency). Good for social/email. Included size: 1024 px.
PDF → Print-ready, preserves vector when possible. Printers love this.
EPS → Industry-standard vector for signage/embroidery/swag (if included).

Tip: Use SVG for design & high-quality web; PNG for transparent placements; JPEG for quick previews; PDF/EPS for professional print.`;

export function ExportModal({ open, onOpenChange, svgMarkup, brandName }: ExportModalProps) {
  const { toast } = useToast();
  const [formats, setFormats] = useState<ExportFormats>({
    svg: true,
    png: true,
    jpeg: false,
    pdf: false,
    eps: false,
    webp: false,
  });
  const [pngSizes, setPngSizes] = useState<PngSizes>({
    256: true,
    512: true,
    1024: true,
  });
  const [isExporting, setIsExporting] = useState(false);

  const handleFormatChange = (format: keyof ExportFormats, checked: boolean) => {
    setFormats(prev => ({ ...prev, [format]: checked }));
  };

  const handlePngSizeChange = (size: keyof PngSizes, checked: boolean) => {
    setPngSizes(prev => ({ ...prev, [size]: checked }));
  };

  const isValid = Object.values(formats).some(v => v);

  const handleExport = async () => {
    if (!isValid) {
      toast({
        title: "No formats selected",
        description: "Please select at least one export format.",
        variant: "destructive",
      });
      return;
    }

    setIsExporting(true);
    const files: ZipFile[] = [];
    const errors: string[] = [];
    const safeBrandName = brandName.replace(/[^a-z0-9]/gi, '_').toLowerCase() || 'logo';

    try {
      // SVG
      if (formats.svg) {
        try {
          const normalizedSvg = normalizeSvg(svgMarkup);
          files.push({
            name: `${safeBrandName}/logo.svg`,
            content: new Blob([normalizedSvg], { type: 'image/svg+xml' }),
          });
        } catch (err) {
          errors.push('SVG');
          console.error('SVG export failed:', err);
        }
      }

      // PNG (multiple sizes)
      if (formats.png) {
        const selectedSizes = Object.entries(pngSizes)
          .filter(([_, enabled]) => enabled)
          .map(([size, _]) => parseInt(size));

        if (selectedSizes.length === 0) {
          toast({
            title: "No PNG sizes selected",
            description: "Please select at least one PNG size.",
            variant: "destructive",
          });
          setIsExporting(false);
          return;
        }

        for (const size of selectedSizes) {
          try {
            const pngBlob = await svgToPng(svgMarkup, size);
            files.push({
              name: `${safeBrandName}/logo-${size}.png`,
              content: pngBlob,
            });
          } catch (err) {
            errors.push(`PNG (${size}px)`);
            console.error(`PNG ${size}px export failed:`, err);
          }
        }
      }

      // JPEG
      if (formats.jpeg) {
        try {
          const pngBlob = await svgToPng(svgMarkup, 1024);
          const jpegBlob = await pngToJpeg(pngBlob);
          files.push({
            name: `${safeBrandName}/logo-1024.jpg`,
            content: jpegBlob,
          });
        } catch (err) {
          errors.push('JPEG');
          console.error('JPEG export failed:', err);
        }
      }

      // PDF
      if (formats.pdf) {
        try {
          const pdfBlob = await svgToPdf(svgMarkup);
          files.push({
            name: `${safeBrandName}/logo.pdf`,
            content: pdfBlob,
          });
        } catch (err) {
          errors.push('PDF');
          console.error('PDF export failed:', err);
        }
      }

      // WEBP
      if (formats.webp) {
        try {
          const pngBlob = await svgToPng(svgMarkup, 1024);
          const webpBlob = await pngToWebp(pngBlob);
          files.push({
            name: `${safeBrandName}/logo.webp`,
            content: webpBlob,
          });
        } catch (err) {
          errors.push('WEBP');
          console.error('WEBP export failed:', err);
        }
      }

      // EPS (graceful skip)
      if (formats.eps) {
        try {
          const epsBlob = await svgToEps(svgMarkup);
          files.push({
            name: `${safeBrandName}/logo.eps`,
            content: epsBlob,
          });
        } catch (err) {
          toast({
            title: "EPS unavailable",
            description: "EPS export requires desktop tools (Illustrator/Inkscape). Other formats will be included.",
            variant: "default",
          });
          console.error('EPS export failed:', err);
        }
      }

      // Add README
      files.push({
        name: `${safeBrandName}/README.txt`,
        content: README_CONTENT,
      });

      // Create ZIP
      if (files.length > 0) {
        await createZip(files, `${safeBrandName}_logo_export.zip`);
        
        if (errors.length > 0) {
          toast({
            title: "Partial export successful",
            description: `Downloaded with ${errors.length} format(s) skipped: ${errors.join(', ')}`,
          });
        } else {
          toast({
            title: "Export successful",
            description: "Your logo files have been downloaded.",
          });
        }
        
        onOpenChange(false);
      } else {
        throw new Error('No files were generated');
      }
    } catch (err) {
      console.error('Export failed:', err);
      toast({
        title: "Export failed",
        description: "An error occurred while exporting your logo. Please try again.",
        variant: "destructive",
      });
    } finally {
      setIsExporting(false);
    }
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-[500px]">
        <DialogHeader>
          <DialogTitle>Export Logo Files</DialogTitle>
          <DialogDescription>
            Choose formats to export. We'll bundle them in one ZIP.
          </DialogDescription>
        </DialogHeader>

        <ScrollArea className="max-h-[60vh] pr-4">
          <div className="space-y-4 py-4">
            {/* SVG */}
            <div className="flex items-start space-x-3">
              <Checkbox
                id="format-svg"
                checked={formats.svg}
                onCheckedChange={(checked) => handleFormatChange('svg', checked as boolean)}
                data-testid="checkbox-format-svg"
              />
              <div className="flex-1 space-y-1">
                <Label htmlFor="format-svg" className="font-medium cursor-pointer">
                  SVG
                </Label>
                <p className="text-sm text-gray-500">
                  Scalable vector; editable in Figma/Illustrator; web-ready
                </p>
              </div>
            </div>

            {/* PNG */}
            <div className="space-y-2">
              <div className="flex items-start space-x-3">
                <Checkbox
                  id="format-png"
                  checked={formats.png}
                  onCheckedChange={(checked) => handleFormatChange('png', checked as boolean)}
                  data-testid="checkbox-format-png"
                />
                <div className="flex-1 space-y-1">
                  <Label htmlFor="format-png" className="font-medium cursor-pointer">
                    PNG
                  </Label>
                  <p className="text-sm text-gray-500">
                    Transparent; web/docs. Sizes: 256/512/1024 px
                  </p>
                </div>
              </div>

              {formats.png && (
                <div className="ml-7 pl-4 border-l-2 border-gray-200 space-y-2">
                  <div className="flex items-center space-x-2">
                    <Checkbox
                      id="png-256"
                      checked={pngSizes[256]}
                      onCheckedChange={(checked) => handlePngSizeChange(256, checked as boolean)}
                      data-testid="checkbox-png-256"
                    />
                    <Label htmlFor="png-256" className="text-sm cursor-pointer">256px</Label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <Checkbox
                      id="png-512"
                      checked={pngSizes[512]}
                      onCheckedChange={(checked) => handlePngSizeChange(512, checked as boolean)}
                      data-testid="checkbox-png-512"
                    />
                    <Label htmlFor="png-512" className="text-sm cursor-pointer">512px</Label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <Checkbox
                      id="png-1024"
                      checked={pngSizes[1024]}
                      onCheckedChange={(checked) => handlePngSizeChange(1024, checked as boolean)}
                      data-testid="checkbox-png-1024"
                    />
                    <Label htmlFor="png-1024" className="text-sm cursor-pointer">1024px</Label>
                  </div>
                </div>
              )}
            </div>

            <Separator />

            {/* JPEG */}
            <div className="flex items-start space-x-3">
              <Checkbox
                id="format-jpeg"
                checked={formats.jpeg}
                onCheckedChange={(checked) => handleFormatChange('jpeg', checked as boolean)}
                data-testid="checkbox-format-jpeg"
              />
              <div className="flex-1 space-y-1">
                <Label htmlFor="format-jpeg" className="font-medium cursor-pointer">
                  JPEG
                </Label>
                <p className="text-sm text-gray-500">
                  Compressed (no alpha); social/email (1024 px)
                </p>
              </div>
            </div>

            {/* PDF */}
            <div className="flex items-start space-x-3">
              <Checkbox
                id="format-pdf"
                checked={formats.pdf}
                onCheckedChange={(checked) => handleFormatChange('pdf', checked as boolean)}
                data-testid="checkbox-format-pdf"
              />
              <div className="flex-1 space-y-1">
                <Label htmlFor="format-pdf" className="font-medium cursor-pointer">
                  PDF
                </Label>
                <p className="text-sm text-gray-500">
                  Print-ready; preserves vector when possible
                </p>
              </div>
            </div>

            {/* EPS */}
            <div className="flex items-start space-x-3">
              <Checkbox
                id="format-eps"
                checked={formats.eps}
                onCheckedChange={(checked) => handleFormatChange('eps', checked as boolean)}
                data-testid="checkbox-format-eps"
              />
              <div className="flex-1 space-y-1">
                <Label htmlFor="format-eps" className="font-medium cursor-pointer">
                  EPS
                </Label>
                <p className="text-sm text-gray-500">
                  Vector for pro print vendors
                </p>
              </div>
            </div>

            <Separator />

            {/* WEBP */}
            <div className="flex items-start space-x-3">
              <Checkbox
                id="format-webp"
                checked={formats.webp}
                onCheckedChange={(checked) => handleFormatChange('webp', checked as boolean)}
                data-testid="checkbox-format-webp"
              />
              <div className="flex-1 space-y-1">
                <Label htmlFor="format-webp" className="font-medium cursor-pointer">
                  WEBP (Optional)
                </Label>
                <p className="text-sm text-gray-500">
                  Modern compressed raster (1024 px)
                </p>
              </div>
            </div>

            {/* Future: Icon-Only Generator hook */}
            {/* 
            Future: Icon-Only Generator (favicons/app icons)
            Requires the editor to maintain separate SVG layers/groups:
            <g id="icon-layer">…</g> and <g id="text-layer">…</g>
            Export pipeline would render only #icon-layer at 16/32/48/180/512 px, etc.
            */}
          </div>
        </ScrollArea>

        <DialogFooter className="flex-col sm:flex-row gap-2">
          <p className="text-xs text-gray-500 flex-1">
            Large exports may take a few seconds.
          </p>
          <div className="flex gap-2">
            <Button
              variant="ghost"
              onClick={() => onOpenChange(false)}
              disabled={isExporting}
              data-testid="button-cancel-export"
            >
              Cancel
            </Button>
            <Button
              onClick={handleExport}
              disabled={!isValid || isExporting}
              data-testid="button-confirm-export"
            >
              {isExporting && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
              {isExporting ? 'Exporting...' : 'Confirm & Download'}
            </Button>
          </div>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
