import { X } from "lucide-react";
import { Button } from "@/components/ui/button";

type PreviewLightboxProps = {
  open: boolean;
  onClose: () => void;
  title?: string;
  imgSrc: string;
  onDownload?: () => void;
  onCustomize?: () => void;
  customizeDisabled?: boolean;
};

export function PreviewLightbox({
  open,
  onClose,
  title = "Preview",
  imgSrc,
  onDownload,
  onCustomize,
  customizeDisabled = false,
}: PreviewLightboxProps) {
  if (!open) return null;

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if (e.key === "Escape") {
      onClose();
    }
  };

  return (
    <div
      role="dialog"
      aria-modal="true"
      aria-label={title}
      className="fixed inset-0 z-50 flex items-center justify-center"
      onKeyDown={handleKeyDown}
      data-testid="preview-lightbox"
    >
      {/* Backdrop */}
      <div
        className="absolute inset-0 bg-black/70"
        onClick={onClose}
        aria-hidden="true"
      />

      {/* Modal */}
      <div className="relative z-10 w-[92vw] max-w-4xl max-h-[90vh] bg-white dark:bg-gray-900 rounded-xl shadow-xl overflow-hidden flex flex-col">
        {/* Header */}
        <div className="flex items-center justify-between px-4 py-3 border-b border-gray-200 dark:border-gray-700">
          <h3 className="text-base font-semibold text-gray-900 dark:text-gray-100">
            {title}
          </h3>
          <button
            className="text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 transition-colors"
            aria-label="Close preview"
            onClick={onClose}
            data-testid="button-close-preview"
          >
            <X className="h-5 w-5" />
          </button>
        </div>

        {/* Content (scroll/pan) */}
        <div className="flex-1 overflow-auto bg-white dark:bg-gray-900 p-4">
          <div className="min-h-[60vh] flex items-center justify-center">
            <img
              src={imgSrc}
              alt={title}
              className="max-w-full h-auto object-contain select-none"
              draggable={false}
              data-testid="preview-image"
            />
          </div>
        </div>

        {/* Footer Actions */}
        <div className="flex items-center justify-between gap-2 px-4 py-3 border-t border-gray-200 dark:border-gray-700 flex-wrap sm:flex-nowrap">
          <Button
            variant="outline"
            onClick={onDownload}
            className="flex-1 sm:flex-initial"
            data-testid="button-download-from-preview"
          >
            Download SVG
          </Button>
          <Button
            onClick={onCustomize}
            disabled={customizeDisabled}
            className="flex-1 sm:flex-initial bg-[#FF8B00] hover:bg-[#e67d00] text-white"
            data-testid="button-customize-from-preview"
          >
            Customize in Composer
          </Button>
        </div>
      </div>
    </div>
  );
}
