import { useState, useEffect } from "react";
import { V2_TEMPLATES, renderV2TemplateMarkdown } from "@/lib/structured-templates.v2";
import { usePlanStoreV2 } from "@/stores/usePlanStore.v2";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle } from "@/components/ui/alert-dialog";
import { Sparkles, RefreshCw, Save, Loader2 } from "lucide-react";
import { aiClient } from "@/services/ai/aiClient";
import { useToast } from "@/hooks/use-toast";

interface StructuredFormV2Props {
  targetSectionId: string;
}

export function StructuredFormV2({ targetSectionId }: StructuredFormV2Props) {
  const [selectedKey, setSelectedKey] = useState<string>(V2_TEMPLATES[0].key);
  const [data, setData] = useState<Record<string, string>>({});
  const [isGenerating, setIsGenerating] = useState(false);
  const [isRegenerating, setIsRegenerating] = useState(false);
  const [hasGenerated, setHasGenerated] = useState(false);
  const [isSeedDraft, setIsSeedDraft] = useState(false);
  const [lastSavedData, setLastSavedData] = useState<Record<string, string> | null>(null);
  const [showDuplicateDialog, setShowDuplicateDialog] = useState(false);
  
  const update = usePlanStoreV2((s) => s.updateSection);
  const sections = usePlanStoreV2((s) => s.sections);
  const businessBrief = usePlanStoreV2((s) => s.businessBrief);
  
  const { toast } = useToast();
  
  const currentSection = sections.find(s => s.id === targetSectionId);
  const template = V2_TEMPLATES.find((t) => t.key === selectedKey)!;

  // Auto-generate when template is selected
  useEffect(() => {
    async function autoGenerate() {
      if (!hasGenerated && !isGenerating) {
        setIsGenerating(true);
        try {
          const result = await aiClient.generateStructuredTemplate(selectedKey, businessBrief);
          
          // Check if we got valid data
          const hasValidData = Object.values(result.data).some(v => v && v.trim() !== "");
          if (!hasValidData) {
            throw new Error("No content was generated");
          }
          
          setData(result.data);
          setIsSeedDraft(result.isSeedDraft || false);
          setHasGenerated(true);
          
          // Show info message if seed draft was used
          if (result.isSeedDraft) {
            toast({
              title: "AI Draft Unavailable",
              description: "Starter content inserted. Edit as needed or try Regenerate for AI-generated content.",
            });
          }
        } catch (error) {
          console.error('Auto-generation failed:', error);
          const errorMessage = error instanceof Error ? error.message : "Unknown error";
          toast({
            title: "AI Generation Failed",
            description: `${errorMessage}. Please fill in the fields manually or try regenerating.`,
            variant: "destructive"
          });
          setHasGenerated(true); // Mark as attempted so user can still use the form
        } finally {
          setIsGenerating(false);
        }
      }
    }
    
    autoGenerate();
  }, [selectedKey, businessBrief, hasGenerated, isGenerating, toast]);

  // Reset when template changes
  const handleTemplateChange = (newKey: string) => {
    setSelectedKey(newKey);
    setData({});
    setHasGenerated(false);
    setIsSeedDraft(false);
    setLastSavedData(null);
  };

  // Regenerate AI content
  async function handleRegenerate() {
    setIsRegenerating(true);
    try {
      const result = await aiClient.generateStructuredTemplate(selectedKey, businessBrief);
      
      // Check if we got valid data
      const hasValidData = Object.values(result.data).some(v => v && v.trim() !== "");
      if (!hasValidData) {
        throw new Error("No content was generated");
      }
      
      setData(result.data);
      setIsSeedDraft(result.isSeedDraft || false);
      
      if (result.isSeedDraft) {
        toast({
          title: "AI Draft Unavailable",
          description: "Starter content inserted. Edit as needed or try again later.",
        });
      } else {
        toast({
          title: "Content Regenerated",
          description: "Fresh AI-generated content is ready for you to review.",
        });
      }
    } catch (error) {
      console.error('Regeneration failed:', error);
      const errorMessage = error instanceof Error ? error.message : "Unknown error";
      toast({
        title: "Regeneration Failed",
        description: `${errorMessage}. Please try again or fill in manually.`,
        variant: "destructive"
      });
    } finally {
      setIsRegenerating(false);
    }
  }

  // Check if data has changed since last save
  const hasDataChanged = () => {
    if (!lastSavedData) return true; // First save, always allow
    
    // Compare current data with last saved data
    const currentKeys = Object.keys(data).sort();
    const savedKeys = Object.keys(lastSavedData).sort();
    
    // Different number of fields
    if (currentKeys.length !== savedKeys.length) return true;
    
    // Compare each field
    for (const key of currentKeys) {
      if (data[key] !== lastSavedData[key]) return true;
    }
    
    return false; // No changes detected
  };

  // Handle save button click with duplicate detection
  function handleSave() {
    if (!hasDataChanged()) {
      // Show duplicate save dialog
      setShowDuplicateDialog(true);
      return;
    }
    
    // Content has changed, save normally
    performSave();
  }

  // Perform the actual save operation
  function performSave() {
    const md = renderV2TemplateMarkdown(template.key, data);
    const currentContent = currentSection?.content || "";
    
    // Look for "Frameworks" heading at the start of a line (supports optional colon and leading #)
    // Matches: ## Frameworks, ### Frameworks:, Frameworks, Frameworks:, etc.
    const frameworksRegex = /^\s*(?:#{1,6}\s*)?Frameworks\s*:?\s*$/im;
    const lines = currentContent.split('\n');
    let insertLineIndex = -1;
    
    // Find the line with the Frameworks heading
    for (let i = 0; i < lines.length; i++) {
      if (frameworksRegex.test(lines[i])) {
        insertLineIndex = i;
        break;
      }
    }
    
    let newContent: string;
    if (insertLineIndex !== -1) {
      // Found the Frameworks heading - insert after it
      lines.splice(insertLineIndex + 1, 0, '', md, '');
      newContent = lines.join('\n');
    } else {
      // No Frameworks heading found - append to end
      newContent = currentContent ? `${currentContent}\n\n${md}` : md;
    }
    
    update(targetSectionId, { content: newContent });
    
    // Store snapshot of saved data
    setLastSavedData({ ...data });
    
    toast({
      title: "Saved to Section",
      description: "Framework content has been added to your Market Analysis section.",
    });
    
    // Close dialog if it was open
    setShowDuplicateDialog(false);
    
    // Reset for next use
    setData({});
    setHasGenerated(false);
    setLastSavedData(null);
  }

  // Handle "Save Anyway" from duplicate dialog
  function handleSaveAnyway() {
    performSave();
  }

  // Handle "Cancel" from duplicate dialog
  function handleCancelSave() {
    setShowDuplicateDialog(false);
  }

  const hasData = Object.values(data).some(value => value && value.trim() !== "");

  return (
    <>
      <Card className="border-accent/20" data-testid="structured-form-v2">
      <CardHeader className="bg-gradient-to-r from-accent/5 to-transparent">
        <CardTitle className="flex items-center gap-2 text-accent">
          <Sparkles className="h-5 w-5" />
          AI Frameworks Builder
        </CardTitle>
        <CardDescription>
          AI pre-fills strategic frameworks. Edit or regenerate, then save to your section.
        </CardDescription>
      </CardHeader>
      
      <CardContent className="space-y-4 pt-6">
        <div className="space-y-2">
          <Label htmlFor="template-select">Choose a framework</Label>
          <Select value={selectedKey} onValueChange={handleTemplateChange}>
            <SelectTrigger data-testid="select-template">
              <SelectValue placeholder="Choose a framework…" />
            </SelectTrigger>
            <SelectContent>
              {V2_TEMPLATES.map((template) => (
                <SelectItem key={template.key} value={template.key} data-testid={`template-option-${template.key}`}>
                  {template.name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
          <p className="text-sm text-muted-foreground">{template.description}</p>
        </div>

        {isGenerating ? (
          <div className="flex items-center justify-center py-12 text-muted-foreground">
            <Loader2 className="h-6 w-6 animate-spin mr-2" />
            <span>Generating AI content...</span>
          </div>
        ) : (
          <>
            <div className="space-y-4">
              {template.fields.map((field) => (
                <div key={field.id} className="space-y-2">
                  <Label htmlFor={field.id}>{field.label}</Label>
                  {field.type === "textarea" ? (
                    <Textarea
                      id={field.id}
                      placeholder={field.placeholder}
                      value={data[field.id] || ""}
                      onChange={(e) => setData((prev) => ({ ...prev, [field.id]: e.target.value }))}
                      rows={3}
                      className="bg-accent/5 border-accent/20 focus:border-accent"
                      data-testid={`textarea-${field.id}`}
                    />
                  ) : field.type === "select" && field.options ? (
                    <Select 
                      value={data[field.id] || ""} 
                      onValueChange={(value) => setData((prev) => ({ ...prev, [field.id]: value }))}
                    >
                      <SelectTrigger data-testid={`select-${field.id}`}>
                        <SelectValue placeholder={field.placeholder} />
                      </SelectTrigger>
                      <SelectContent>
                        {field.options.map((option) => (
                          <SelectItem key={option} value={option}>
                            {option}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  ) : (
                    <Input
                      id={field.id}
                      type={field.type}
                      placeholder={field.placeholder}
                      value={data[field.id] || ""}
                      onChange={(e) => setData((prev) => ({ ...prev, [field.id]: e.target.value }))}
                      className="bg-accent/5 border-accent/20 focus:border-accent"
                      data-testid={`input-${field.id}`}
                    />
                  )}
                </div>
              ))}
            </div>

            <div className="flex gap-2 pt-4 border-t">
              <Button
                onClick={handleRegenerate}
                disabled={isRegenerating || !hasGenerated}
                variant="outline"
                className="flex-1"
                data-testid="button-regenerate"
              >
                {isRegenerating ? (
                  <>
                    <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                    Regenerating...
                  </>
                ) : (
                  <>
                    <RefreshCw className="h-4 w-4 mr-2" />
                    Regenerate
                  </>
                )}
              </Button>
              
              <Button
                onClick={handleSave}
                disabled={!hasData}
                className="flex-1 bg-accent hover:bg-accent/90"
                data-testid="button-save-template"
              >
                <Save className="h-4 w-4 mr-2" />
                Save to Section
              </Button>
            </div>
          </>
        )}
      </CardContent>
      </Card>
      
      <AlertDialog open={showDuplicateDialog} onOpenChange={setShowDuplicateDialog}>
        <AlertDialogContent data-testid="dialog-duplicate-save">
          <AlertDialogHeader>
            <AlertDialogTitle>No changes since last save</AlertDialogTitle>
            <AlertDialogDescription>
              This section hasn't changed since your last save. Do you still want to save it again?
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel onClick={handleCancelSave} data-testid="button-cancel-duplicate">
              Cancel
            </AlertDialogCancel>
            <AlertDialogAction 
              onClick={handleSaveAnyway} 
              className="bg-accent hover:bg-accent/90"
              data-testid="button-save-anyway"
            >
              Save Anyway
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </>
  );
}
