import { useState } from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Form, FormField, FormItem, FormLabel, FormControl, FormMessage } from '@/components/ui/form';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { FileText, Eye, Download, Layout } from 'lucide-react';
import { STRUCTURED_TEMPLATES, renderTemplateToMarkdown } from '@/lib/structured-templates';
import { usePlanStore } from '@/stores/usePlanStore';
import type { StructuredTemplate, SectionId } from '@shared/plan-types';

interface StructuredFormProps {
  targetSectionId?: string;
  onClose?: () => void;
}

export function StructuredForm({ targetSectionId, onClose }: StructuredFormProps) {
  const [selectedTemplate, setSelectedTemplate] = useState<StructuredTemplate | null>(null);
  const [activeTab, setActiveTab] = useState<'form' | 'preview'>('form');
  const [previewContent, setPreviewContent] = useState('');
  
  const { updateSection, sections } = usePlanStore();
  
  // Dynamic schema based on selected template
  const createFormSchema = (template: StructuredTemplate | null) => {
    if (!template) return z.object({});
    
    const schemaFields: Record<string, z.ZodString> = {};
    template.fields.forEach(field => {
      schemaFields[field.id] = field.type === 'text' 
        ? z.string().min(1, `${field.label} is required`)
        : z.string().optional().or(z.literal(''));
    });
    
    return z.object(schemaFields);
  };

  const formSchema = createFormSchema(selectedTemplate);
  type FormData = z.infer<typeof formSchema>;

  const form = useForm<FormData>({
    resolver: zodResolver(formSchema),
    defaultValues: selectedTemplate?.fields.reduce((acc, field) => {
      acc[field.id] = '';
      return acc;
    }, {} as Record<string, string>) || {},
  });

  const watchedValues = form.watch();

  // Update preview when form values change
  useState(() => {
    if (selectedTemplate) {
      const preview = renderTemplateToMarkdown(selectedTemplate.key, watchedValues);
      setPreviewContent(preview);
    }
  }, [selectedTemplate, watchedValues]);

  const handleTemplateSelect = (template: StructuredTemplate) => {
    setSelectedTemplate(template);
    
    // Reset form with new default values
    const defaultValues = template.fields.reduce((acc, field) => {
      acc[field.id] = '';
      return acc;
    }, {} as Record<string, string>);
    
    form.reset(defaultValues);
    setActiveTab('form');
  };

  const handleSubmit = (data: FormData) => {
    if (!selectedTemplate || !targetSectionId) return;
    
    const markdown = renderTemplateToMarkdown(selectedTemplate.key, data);
    updateSection(targetSectionId, { content: markdown });
    
    // Close the form
    onClose?.();
  };

  const handlePreview = () => {
    if (selectedTemplate) {
      const preview = renderTemplateToMarkdown(selectedTemplate.key, watchedValues);
      setPreviewContent(preview);
      setActiveTab('preview');
    }
  };

  const targetSection = targetSectionId 
    ? sections.find(s => s.id === targetSectionId)
    : null;

  return (
    <Card className="w-full" data-testid="structured-form">
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <FileText className="w-5 h-5" />
          Structured Templates
          {targetSection && (
            <Badge variant="outline" className="ml-auto">
              {targetSection.title}
            </Badge>
          )}
        </CardTitle>
      </CardHeader>

      <CardContent className="space-y-6">
        {/* Template Selection */}
        {!selectedTemplate && (
          <div className="space-y-4" data-testid="template-selection">
            <h3 className="text-sm font-medium">Choose a Template</h3>
            <div className="grid gap-3">
              {STRUCTURED_TEMPLATES.map((template) => (
                <Card
                  key={template.key}
                  className="cursor-pointer hover:shadow-md transition-shadow border-2 border-transparent hover:border-primary/20"
                  onClick={() => handleTemplateSelect(template)}
                  data-testid={`template-card-${template.key}`}
                >
                  <CardContent className="p-4">
                    <h4 className="font-medium text-sm mb-1">{template.name}</h4>
                    <p className="text-xs text-muted-foreground">{template.description}</p>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        )}

        {/* Selected Template Form */}
        {selectedTemplate && (
          <div className="space-y-4" data-testid="template-form">
            <div className="flex items-center justify-between">
              <div>
                <h3 className="text-sm font-medium">{selectedTemplate.name}</h3>
                <p className="text-xs text-muted-foreground">{selectedTemplate.description}</p>
              </div>
              <Button
                variant="outline"
                size="sm"
                onClick={() => {
                  setSelectedTemplate(null);
                  form.reset({});
                }}
                data-testid="button-change-template"
              >
                Change Template
              </Button>
            </div>

            <Tabs value={activeTab} onValueChange={(value) => setActiveTab(value as 'form' | 'preview')}>
              <TabsList className="grid w-full grid-cols-2">
                <TabsTrigger value="form" data-testid="tab-form">Form</TabsTrigger>
                <TabsTrigger value="preview" data-testid="tab-preview">Preview</TabsTrigger>
              </TabsList>

              <TabsContent value="form" className="mt-4">
                <Form {...form}>
                  <form onSubmit={form.handleSubmit(handleSubmit)} className="space-y-4">
                    {selectedTemplate.fields.map((field) => (
                      <FormField
                        key={field.id}
                        control={form.control}
                        name={field.id}
                        render={({ field: formField }) => (
                          <FormItem>
                            <FormLabel>{field.label}</FormLabel>
                            <FormControl>
                              {field.type === 'textarea' ? (
                                <Textarea
                                  {...formField}
                                  placeholder={field.placeholder}
                                  rows={4}
                                  data-testid={`field-${field.id}`}
                                />
                              ) : field.type === 'select' ? (
                                <select 
                                  {...formField}
                                  className="flex h-10 w-full rounded-md border border-input bg-background px-3 py-2 text-sm ring-offset-background focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2"
                                  data-testid={`field-${field.id}`}
                                >
                                  <option value="">Select...</option>
                                  {field.options?.map((option) => (
                                    <option key={option} value={option}>
                                      {option}
                                    </option>
                                  ))}
                                </select>
                              ) : (
                                <Input
                                  {...formField}
                                  type={field.type}
                                  placeholder={field.placeholder}
                                  data-testid={`field-${field.id}`}
                                />
                              )}
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                    ))}

                    <div className="flex gap-3 pt-4">
                      <Button
                        type="button"
                        variant="outline"
                        onClick={handlePreview}
                        className="flex-1"
                        data-testid="button-preview"
                      >
                        <Eye className="w-4 h-4 mr-2" />
                        Preview
                      </Button>
                      <Button
                        type="submit"
                        disabled={!targetSectionId}
                        className="flex-1"
                        data-testid="button-insert"
                      >
                        <Download className="w-4 h-4 mr-2" />
                        Insert into Section
                      </Button>
                    </div>
                  </form>
                </Form>
              </TabsContent>

              <TabsContent value="preview" className="mt-4">
                <div className="space-y-4">
                  <div 
                    className="prose prose-sm max-w-none p-4 border rounded-lg bg-muted/20"
                    data-testid="template-preview"
                  >
                    {previewContent ? (
                      <div dangerouslySetInnerHTML={{ 
                        __html: previewContent.replace(/\n/g, '<br>').replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>') 
                      }} />
                    ) : (
                      <p className="text-muted-foreground text-sm">Fill out the form to see preview...</p>
                    )}
                  </div>

                  <div className="flex gap-3">
                    <Button
                      variant="outline"
                      onClick={() => setActiveTab('form')}
                      className="flex-1"
                      data-testid="button-edit"
                    >
                      Edit
                    </Button>
                    <Button
                      onClick={form.handleSubmit(handleSubmit)}
                      disabled={!targetSectionId || !previewContent.trim()}
                      className="flex-1"
                      data-testid="button-insert-preview"
                    >
                      <Download className="w-4 h-4 mr-2" />
                      Insert into Section
                    </Button>
                  </div>
                </div>
              </TabsContent>
            </Tabs>
          </div>
        )}

        {/* No Target Section Warning */}
        {!targetSectionId && (
          <div className="p-3 bg-amber-50 border border-amber-200 rounded-lg text-amber-800 text-sm">
            Select a section from the outline to insert template content.
          </div>
        )}
      </CardContent>
    </Card>
  );
}