import { useState, useRef } from 'react';
import { useSortable } from '@dnd-kit/sortable';
import { CSS } from '@dnd-kit/utilities';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { GripVertical, Copy, Trash2, Edit3 } from 'lucide-react';
import { cn } from '@/lib/utils';
import type { PlanSection } from '@shared/plan-types';

interface SortableItemProps {
  section: PlanSection;
  isSelected: boolean;
  onSelect: (id: string) => void;
  onUpdateTitle: (id: string, title: string) => void;
  onDuplicate: (id: string) => void;
  onDelete: (id: string) => void;
}

export function SortableItem({
  section,
  isSelected,
  onSelect,
  onUpdateTitle,
  onDuplicate,
  onDelete,
}: SortableItemProps) {
  const [isEditing, setIsEditing] = useState(false);
  const [editTitle, setEditTitle] = useState(section.title);
  const inputRef = useRef<HTMLInputElement>(null);

  const {
    attributes,
    listeners,
    setNodeRef,
    transform,
    transition,
    isDragging,
  } = useSortable({ id: section.id });

  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
  };

  const handleTitleEdit = () => {
    setIsEditing(true);
    setEditTitle(section.title);
    setTimeout(() => inputRef.current?.focus(), 0);
  };

  const handleTitleSave = () => {
    const trimmed = editTitle.trim();
    if (trimmed && trimmed !== section.title) {
      onUpdateTitle(section.id, trimmed);
    } else {
      setEditTitle(section.title);
    }
    setIsEditing(false);
  };

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter') {
      e.preventDefault();
      handleTitleSave();
    } else if (e.key === 'Escape') {
      setEditTitle(section.title);
      setIsEditing(false);
    }
  };

  return (
    <div
      ref={setNodeRef}
      style={style}
      className={cn(
        "group relative bg-white border rounded-lg p-3 transition-all",
        isSelected && "border-primary bg-primary/5",
        isDragging && "opacity-50 shadow-lg",
        "hover:shadow-md"
      )}
      data-testid={`sortable-item-${section.id}`}
    >
      {/* Drag Handle */}
      <div
        {...attributes}
        {...listeners}
        className="absolute left-1 top-1/2 -translate-y-1/2 cursor-grab active:cursor-grabbing opacity-0 group-hover:opacity-100 transition-opacity"
        data-testid={`drag-handle-${section.id}`}
      >
        <GripVertical className="w-4 h-4 text-muted-foreground" />
      </div>

      {/* Content */}
      <div className="pl-6 pr-24">
        {isEditing ? (
          <Input
            ref={inputRef}
            value={editTitle}
            onChange={(e) => setEditTitle(e.target.value)}
            onBlur={handleTitleSave}
            onKeyDown={handleKeyDown}
            className="text-sm font-medium"
            data-testid={`input-edit-title-${section.id}`}
          />
        ) : (
          <button
            onClick={() => onSelect(section.id)}
            className="text-left w-full text-sm font-medium text-foreground hover:text-primary transition-colors"
            data-testid={`button-select-section-${section.id}`}
          >
            {section.title}
          </button>
        )}
        
        {section.content && (
          <p className="text-xs text-muted-foreground mt-1 line-clamp-2">
            {section.content.slice(0, 100)}...
          </p>
        )}
      </div>

      {/* Actions */}
      <div className="absolute right-2 top-1/2 -translate-y-1/2 flex gap-1 opacity-0 group-hover:opacity-100 transition-opacity">
        <Button
          size="sm"
          variant="ghost"
          onClick={handleTitleEdit}
          className="h-7 w-7 p-0"
          data-testid={`button-edit-${section.id}`}
        >
          <Edit3 className="w-3 h-3" />
        </Button>
        <Button
          size="sm"
          variant="ghost"
          onClick={() => onDuplicate(section.id)}
          className="h-7 w-7 p-0"
          data-testid={`button-duplicate-${section.id}`}
        >
          <Copy className="w-3 h-3" />
        </Button>
        <Button
          size="sm"
          variant="ghost"
          onClick={() => onDelete(section.id)}
          className="h-7 w-7 p-0 text-destructive hover:text-destructive"
          data-testid={`button-delete-${section.id}`}
        >
          <Trash2 className="w-3 h-3" />
        </Button>
      </div>
    </div>
  );
}