import { DndContext, closestCenter, PointerSensor, useSensor, useSensors } from "@dnd-kit/core";
import { arrayMove, SortableContext, verticalListSortingStrategy } from "@dnd-kit/sortable";
import { CSS } from "@dnd-kit/utilities";
import { useSortable } from "@dnd-kit/sortable";
import { usePlanStoreV2 } from "@/stores/usePlanStore.v2";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { GripVertical, Plus, Trash2, Eye, EyeOff, FileText } from "lucide-react";
import { useState } from "react";
import { useLocation } from "wouter";
import checklistIcon from "@assets/checklist.svg";

interface RowProps {
  id: string;
  title: string;
  isOptional?: boolean;
  isEnabled?: boolean;
  parentId?: string;
}

function Row({ id, title, isOptional, isEnabled, parentId, selectedSectionId, onSectionSelect }: RowProps & { selectedSectionId?: string; onSectionSelect?: (id: string) => void }) {
  const { attributes, listeners, setNodeRef, transform, transition, isDragging } = useSortable({ id });
  const [isEditing, setIsEditing] = useState(false);
  const [editValue, setEditValue] = useState(title);
  
  const update = usePlanStoreV2((s) => s.updateSection);
  const updateChild = usePlanStoreV2((s) => s.updateChildSection);
  const remove = usePlanStoreV2((s) => s.removeSection);
  const removeChild = usePlanStoreV2((s) => s.removeChildSection);
  const addChild = usePlanStoreV2((s) => s.addChildSection);
  const toggleEnabled = usePlanStoreV2((s) => s.toggleSectionEnabled);

  const style: React.CSSProperties = { 
    transform: CSS.Transform.toString(transform), 
    transition, 
    opacity: isDragging ? 0.6 : isEnabled === false ? 0.5 : 1 
  };

  const onRename = (val: string) => {
    if (parentId) {
      updateChild(parentId, id, { title: val });
    } else {
      update(id, { title: val });
    }
  };

  const onDelete = () => {
    if (parentId) {
      removeChild(parentId, id);
    } else {
      remove(id);
    }
  };

  const handleSave = () => {
    onRename(editValue);
    setIsEditing(false);
  };

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter') {
      handleSave();
    } else if (e.key === 'Escape') {
      setEditValue(title);
      setIsEditing(false);
    }
  };

  return (
    <div 
      ref={setNodeRef} 
      style={style} 
      className={`group flex items-center justify-between rounded-lg border p-3 shadow-sm ${
        isEnabled === false 
          ? 'bg-gray-100 dark:bg-gray-900 border-gray-300 dark:border-gray-700' 
          : 'bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-700'
      }`}
      data-testid={`outline-item-${id}`}
    >
      <div className="flex items-center gap-3 flex-1">
        <button 
          {...attributes} 
          {...listeners} 
          className="cursor-grab text-gray-400 hover:text-gray-600 dark:text-gray-600 dark:hover:text-gray-400"
          data-testid={`drag-handle-${id}`}
        >
          <GripVertical className="h-4 w-4" />
        </button>
        
        {isEnabled === false && (
          <EyeOff className="h-4 w-4 text-gray-400" data-testid={`icon-disabled-${id}`} />
        )}
        
        {isEditing ? (
          <Input
            value={editValue}
            onChange={(e) => setEditValue(e.target.value)}
            onBlur={handleSave}
            onKeyDown={handleKeyDown}
            className="flex-1"
            autoFocus
            data-testid={`input-edit-${id}`}
          />
        ) : (
          <div className="flex items-center gap-2 flex-1">
            <span 
              className={`cursor-pointer text-xs font-medium ${
                isEnabled === false 
                  ? 'text-gray-400 dark:text-gray-600 line-through' 
                  : 'text-gray-900 dark:text-gray-100'
              } ${
                selectedSectionId === id ? 'bg-blue-100 dark:bg-blue-900 px-2 py-1 rounded' : ''
              }`}
              onClick={() => onSectionSelect?.(id)}
              onDoubleClick={() => setIsEditing(true)}
              data-testid={`text-title-${id}`}
            >
              {title}
            </span>
            {isOptional && (
              <Badge variant="secondary" className="text-xs" data-testid={`badge-optional-${id}`}>
                Optional
              </Badge>
            )}
          </div>
        )}
      </div>
      
      <div className="flex gap-1 opacity-0 group-hover:opacity-100 transition-opacity">
        {isOptional && !parentId && (
          <Button
            variant="ghost"
            size="sm"
            onClick={() => toggleEnabled(id)}
            className="h-6 px-2 text-xs"
            data-testid={`button-toggle-${id}`}
            title={isEnabled ? "Hide from preview/export" : "Show in preview/export"}
          >
            {isEnabled ? <Eye className="h-3 w-3" /> : <EyeOff className="h-3 w-3" />}
          </Button>
        )}
        {!parentId && (
          <Button
            variant="ghost"
            size="sm"
            onClick={() => addChild(id)}
            className="h-6 px-2 text-xs"
            data-testid={`button-add-subsection-${id}`}
          >
            <Plus className="h-3 w-3 mr-1" />
            Add
          </Button>
        )}
        <Button
          variant="ghost"
          size="sm"
          onClick={onDelete}
          className="h-6 w-6 p-1 text-red-600 hover:text-red-700 dark:text-red-400 dark:hover:text-red-300"
          data-testid={`button-delete-${id}`}
        >
          <Trash2 className="h-3 w-3" />
        </Button>
      </div>
    </div>
  );
}

interface OutlineEditorV2Props {
  selectedSectionId?: string;
  onSectionSelect?: (id: string) => void;
}

export function OutlineEditorV2({ selectedSectionId, onSectionSelect }: OutlineEditorV2Props = {}) {
  const sections = usePlanStoreV2((s) => s.sections);
  const reorder = usePlanStoreV2((s) => s.reorderSections);
  const reorderChild = usePlanStoreV2((s) => s.reorderChildSections);
  const addSection = usePlanStoreV2((s) => s.addSection);
  const sensors = useSensors(useSensor(PointerSensor));

  function onDragEndTop(event: any) {
    const { active, over } = event;
    if (!over || active.id === over.id) return;
    
    const ids = sections.map((s) => s.id);
    const oldIndex = ids.indexOf(active.id);
    const newIndex = ids.indexOf(over.id);
    reorder(arrayMove(ids, oldIndex, newIndex));
  }

  function ChildList({ parentId, selectedSectionId, onSectionSelect }: { parentId: string; selectedSectionId?: string; onSectionSelect?: (id: string) => void }) {
    const parent = sections.find((x) => x.id === parentId);
    const children = parent?.children || [];

    function onDragEndChild(event: any) {
      const { active, over } = event;
      if (!over || active.id === over.id) return;
      
      const ids = children.map((c) => c.id);
      const oldIndex = ids.indexOf(active.id);
      const newIndex = ids.indexOf(over.id);
      reorderChild(parentId, arrayMove(ids, oldIndex, newIndex));
    }

    if (children.length === 0) return null;

    return (
      <div className="mt-3 ml-7" data-testid={`child-list-${parentId}`}>
        <DndContext sensors={sensors} collisionDetection={closestCenter} onDragEnd={onDragEndChild}>
          <SortableContext items={children.map((c) => c.id)} strategy={verticalListSortingStrategy}>
            <div className="space-y-2">
              {children.map((c) => (
                <Row key={c.id} id={c.id} title={c.title} parentId={parentId} selectedSectionId={selectedSectionId} onSectionSelect={onSectionSelect} />
              ))}
            </div>
          </SortableContext>
        </DndContext>
      </div>
    );
  }

  const [, setLocation] = useLocation();

  return (
    <div className="space-y-4" data-testid="outline-editor-v2">
      <div className="flex flex-col gap-2">
        <h2 className="text-lg font-semibold text-gray-900 dark:text-gray-100 flex items-center gap-2">
          <img src={checklistIcon} alt="" className="w-5 h-5" />
          Business Plan Outline
        </h2>
        <div className="flex gap-2">
          <Button
            onClick={() => addSection()}
            size="sm"
            data-testid="button-add-section"
            className="flex-1"
          >
            <Plus className="h-4 w-4 mr-1" />
            Add Section
          </Button>
          <Button
            onClick={() => setLocation('/business-assets/templates/business-plan')}
            size="sm"
            data-testid="button-use-template"
            className="flex-1 bg-accent hover:bg-accent/90 text-white"
          >
            <FileText className="h-4 w-4 mr-1" />
            Use Template
          </Button>
        </div>
      </div>

      <div className="space-y-3">
        <DndContext sensors={sensors} collisionDetection={closestCenter} onDragEnd={onDragEndTop}>
          <SortableContext items={sections.map((s) => s.id)} strategy={verticalListSortingStrategy}>
            {sections.map((s) => (
              <div key={s.id} className="rounded-lg border bg-gray-50 dark:bg-gray-900 p-4" data-testid={`section-container-${s.id}`}>
                <Row 
                  id={s.id} 
                  title={s.title} 
                  isOptional={s.isOptional} 
                  isEnabled={s.isEnabled} 
                  selectedSectionId={selectedSectionId} 
                  onSectionSelect={onSectionSelect} 
                />
                <ChildList parentId={s.id} selectedSectionId={selectedSectionId} onSectionSelect={onSectionSelect} />
              </div>
            ))}
          </SortableContext>
        </DndContext>
      </div>
    </div>
  );
}