import { useState } from 'react';
import {
  DndContext,
  closestCenter,
  KeyboardSensor,
  PointerSensor,
  useSensor,
  useSensors,
  DragEndEvent,
} from '@dnd-kit/core';
import {
  arrayMove,
  SortableContext,
  sortableKeyboardCoordinates,
  verticalListSortingStrategy,
} from '@dnd-kit/sortable';
import {
  restrictToVerticalAxis,
  restrictToParentElement,
} from '@dnd-kit/modifiers';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Plus, FileText, RotateCcw } from 'lucide-react';
import { usePlanStore } from '@/stores/usePlanStore';
import { SortableItem } from './SortableItem';
import type { SectionId } from '@shared/plan-types';

interface OutlineEditorProps {
  selectedSectionId?: string;
  onSectionSelect: (id: string) => void;
}

export function OutlineEditor({ selectedSectionId, onSectionSelect }: OutlineEditorProps) {
  const [newSectionTitle, setNewSectionTitle] = useState('');
  
  const {
    title,
    sections,
    setPlanTitle,
    addSection,
    removeSection,
    reorderSections,
    updateSection,
    duplicateSection,
    resetToDefault,
  } = usePlanStore();

  const sensors = useSensors(
    useSensor(PointerSensor, {
      activationConstraint: {
        distance: 8,
      },
    }),
    useSensor(KeyboardSensor, {
      coordinateGetter: sortableKeyboardCoordinates,
    })
  );

  const handleDragEnd = (event: DragEndEvent) => {
    const { active, over } = event;

    if (over && active.id !== over.id) {
      const activeIndex = sections.findIndex((section) => section.id === active.id);
      const overIndex = sections.findIndex((section) => section.id === over.id);
      
      const reorderedIds = arrayMove(sections, activeIndex, overIndex).map(s => s.id);
      reorderSections(reorderedIds);
    }
  };

  const handleAddSection = () => {
    if (newSectionTitle.trim()) {
      const newId = addSection('custom', newSectionTitle.trim());
      setNewSectionTitle('');
      onSectionSelect(newId);
    }
  };

  const handleUpdateTitle = (id: SectionId, title: string) => {
    updateSection(id, { title });
  };

  const handleDuplicate = (id: SectionId) => {
    const newId = duplicateSection(id);
    if (newId) {
      onSectionSelect(newId);
    }
  };

  const handleDelete = (id: SectionId) => {
    removeSection(id);
    if (selectedSectionId === id && sections.length > 1) {
      // Select the first remaining section
      const remainingSections = sections.filter(s => s.id !== id);
      if (remainingSections.length > 0) {
        onSectionSelect(remainingSections[0].id);
      }
    }
  };

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter') {
      e.preventDefault();
      handleAddSection();
    }
  };

  return (
    <Card className="h-full flex flex-col" data-testid="outline-editor">
      <CardHeader className="pb-4">
        <div className="space-y-3">
          <div>
            <CardTitle className="text-lg mb-2">Business Plan</CardTitle>
            <Input
              value={title}
              onChange={(e) => setPlanTitle(e.target.value)}
              className="font-medium"
              placeholder="Enter plan title..."
              data-testid="input-plan-title"
            />
          </div>
          
          <div className="flex gap-2">
            <Input
              value={newSectionTitle}
              onChange={(e) => setNewSectionTitle(e.target.value)}
              onKeyDown={handleKeyDown}
              placeholder="New section name..."
              className="flex-1"
              data-testid="input-new-section"
            />
            <Button
              onClick={handleAddSection}
              disabled={!newSectionTitle.trim()}
              size="sm"
              data-testid="button-add-section"
            >
              <Plus className="w-4 h-4" />
            </Button>
          </div>

          <div className="flex gap-2">
            <Button
              variant="outline"
              size="sm"
              onClick={resetToDefault}
              className="flex-1"
              data-testid="button-reset-default"
            >
              <RotateCcw className="w-4 h-4 mr-1" />
              Reset
            </Button>
          </div>
        </div>
      </CardHeader>

      <CardContent className="flex-1 overflow-auto">
        <div className="space-y-2">
          {sections.length === 0 ? (
            <div className="text-center py-8 text-muted-foreground" data-testid="empty-state">
              <FileText className="w-8 h-8 mx-auto mb-2 opacity-50" />
              <p className="text-sm">No sections yet</p>
              <p className="text-xs">Add your first section above</p>
            </div>
          ) : (
            <DndContext
              sensors={sensors}
              collisionDetection={closestCenter}
              onDragEnd={handleDragEnd}
              modifiers={[restrictToVerticalAxis, restrictToParentElement]}
            >
              <SortableContext
                items={sections.map(s => s.id)}
                strategy={verticalListSortingStrategy}
              >
                <div className="space-y-2" data-testid="sortable-list">
                  {sections.map((section) => (
                    <SortableItem
                      key={section.id}
                      section={section}
                      isSelected={selectedSectionId === section.id}
                      onSelect={onSectionSelect}
                      onUpdateTitle={handleUpdateTitle}
                      onDuplicate={handleDuplicate}
                      onDelete={handleDelete}
                    />
                  ))}
                </div>
              </SortableContext>
            </DndContext>
          )}
        </div>
      </CardContent>
    </Card>
  );
}