import { useState } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { X, ZoomIn, ZoomOut, Grid3x3, List } from 'lucide-react';

interface BusinessPlanPreviewDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  name: string;
  previewPages: string[];
  docxUrl?: string;
}

export function BusinessPlanPreviewDialog({
  open,
  onOpenChange,
  name,
  previewPages,
  docxUrl
}: BusinessPlanPreviewDialogProps) {
  const [viewMode, setViewMode] = useState<'scroll' | 'grid'>('grid');
  const [selectedPage, setSelectedPage] = useState<number | null>(null);
  const [zoom, setZoom] = useState(100);

  const handleZoomIn = () => setZoom(prev => Math.min(prev + 25, 200));
  const handleZoomOut = () => setZoom(prev => Math.max(prev - 25, 50));

  const hasPageImages = previewPages && previewPages.length > 0;
  const showDocumentViewer = !hasPageImages && docxUrl;
  const isPdf = docxUrl?.toLowerCase().includes('.pdf');

  // Use live proxy for PDFs, Google Docs viewer for DOCX
  const viewerUrl = showDocumentViewer
    ? (isPdf 
        ? `/api/preview/pdf?u=${encodeURIComponent(docxUrl!)}&filename=${encodeURIComponent(name || 'template.pdf')}`
        : `https://docs.google.com/gview?embedded=1&url=${encodeURIComponent(docxUrl!)}`)
    : null;

  return (
    <>
      <Dialog open={open && selectedPage === null} onOpenChange={onOpenChange}>
        <DialogContent className="max-w-6xl max-h-[90vh] overflow-hidden flex flex-col">
          <DialogHeader className="flex-shrink-0">
            <div className="flex items-center justify-between">
              <DialogTitle className="text-xl font-bold">{name} Preview</DialogTitle>
              {hasPageImages && (
                <div className="flex items-center gap-2">
                  <Button
                    variant={viewMode === 'grid' ? 'default' : 'outline'}
                    size="sm"
                    onClick={() => setViewMode('grid')}
                    data-testid="button-grid-view"
                  >
                    <Grid3x3 className="h-4 w-4 mr-1" />
                    Grid
                  </Button>
                  <Button
                    variant={viewMode === 'scroll' ? 'default' : 'outline'}
                    size="sm"
                    onClick={() => setViewMode('scroll')}
                    data-testid="button-scroll-view"
                  >
                    <List className="h-4 w-4 mr-1" />
                    Scroll
                  </Button>
                </div>
              )}
            </div>
          </DialogHeader>

          <div className="flex-1 overflow-auto">
            {showDocumentViewer ? (
              <iframe
                src={viewerUrl!}
                style={{ width: '100%', height: '80vh', border: 0 }}
                sandbox="allow-same-origin allow-scripts allow-popups"
                referrerPolicy="no-referrer"
                title={`${name} Document Preview`}
                data-testid="document-viewer-iframe"
              />
            ) : hasPageImages ? (
              viewMode === 'grid' ? (
                <div className="grid grid-cols-2 md:grid-cols-3 gap-4 p-4">
                  {previewPages.map((page, index) => (
                    <div
                      key={index}
                      className="border rounded-lg overflow-hidden cursor-pointer hover:shadow-lg transition-shadow bg-white"
                      onClick={() => setSelectedPage(index)}
                      data-testid={`preview-thumbnail-${index}`}
                    >
                      <img
                        src={page}
                        alt={`Page ${index + 1}`}
                        className="w-full h-auto object-contain"
                      />
                      <div className="p-2 text-center text-sm text-gray-600 bg-gray-50">
                        Page {index + 1}
                      </div>
                    </div>
                  ))}
                </div>
              ) : (
                <div className="space-y-6 p-4">
                  {previewPages.map((page, index) => (
                    <div key={index} className="border rounded-lg overflow-hidden bg-white">
                      <div className="p-2 bg-gray-50 text-sm font-medium text-gray-700">
                        Page {index + 1}
                      </div>
                      <img
                        src={page}
                        alt={`Page ${index + 1}`}
                        className="w-full h-auto object-contain cursor-pointer"
                        onClick={() => setSelectedPage(index)}
                        data-testid={`preview-page-${index}`}
                      />
                    </div>
                  ))}
                </div>
              )
            ) : (
              <div className="p-8 text-center text-muted-foreground">
                No preview available
              </div>
            )}
          </div>

          {hasPageImages && (
            <div className="flex-shrink-0 border-t pt-3 text-sm text-muted-foreground text-center">
              {previewPages.length} pages • Click any page to view full size
            </div>
          )}
        </DialogContent>
      </Dialog>

      {/* Full-size page viewer */}
      <Dialog open={selectedPage !== null} onOpenChange={(open) => !open && setSelectedPage(null)}>
        <DialogContent className="max-w-7xl max-h-[95vh] overflow-hidden flex flex-col">
          <DialogHeader className="flex-shrink-0">
            <div className="flex items-center justify-between">
              <DialogTitle className="text-lg font-semibold">
                Page {selectedPage !== null ? selectedPage + 1 : 0} of {previewPages.length}
              </DialogTitle>
              <div className="flex items-center gap-2">
                <Button
                  variant="outline"
                  size="sm"
                  onClick={handleZoomOut}
                  disabled={zoom <= 50}
                  data-testid="button-zoom-out"
                >
                  <ZoomOut className="h-4 w-4" />
                </Button>
                <span className="text-sm font-medium w-12 text-center">{zoom}%</span>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={handleZoomIn}
                  disabled={zoom >= 200}
                  data-testid="button-zoom-in"
                >
                  <ZoomIn className="h-4 w-4" />
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => setSelectedPage(null)}
                  data-testid="button-close-fullsize"
                >
                  <X className="h-4 w-4" />
                </Button>
              </div>
            </div>
          </DialogHeader>

          <div className="flex-1 overflow-auto bg-gray-100 flex items-center justify-center p-4">
            {selectedPage !== null && (
              <img
                src={previewPages[selectedPage]}
                alt={`Page ${selectedPage + 1}`}
                style={{ width: `${zoom}%`, maxWidth: 'none' }}
                className="shadow-2xl"
                data-testid="fullsize-preview-image"
              />
            )}
          </div>

          <div className="flex-shrink-0 border-t pt-3 flex items-center justify-between">
            <Button
              variant="outline"
              onClick={() => setSelectedPage(prev => Math.max((prev ?? 0) - 1, 0))}
              disabled={selectedPage === 0}
              data-testid="button-previous-page"
            >
              Previous
            </Button>
            <span className="text-sm text-muted-foreground">
              Page {selectedPage !== null ? selectedPage + 1 : 0} of {previewPages.length}
            </span>
            <Button
              variant="outline"
              onClick={() => setSelectedPage(prev => Math.min((prev ?? 0) + 1, previewPages.length - 1))}
              disabled={selectedPage === previewPages.length - 1}
              data-testid="button-next-page"
            >
              Next
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    </>
  );
}
