import { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Button } from '@/components/ui/button';
import { ChevronDown, ChevronUp, Briefcase, X } from 'lucide-react';
import { usePlanStoreV2 } from '@/stores/usePlanStore.v2';
import type { BusinessBrief as BusinessBriefType } from '@shared/plan-types';

export function BusinessBrief() {
  const [isExpanded, setIsExpanded] = useState(false);
  const { businessBrief = {}, updateBusinessBrief, title, setPlanTitle, planTitleManuallyEdited } = usePlanStoreV2();

  const handleFieldChange = (field: keyof BusinessBriefType, value: string) => {
    updateBusinessBrief({ [field]: value });
  };

  const handlePlanTitleChange = (value: string) => {
    setPlanTitle(value);
  };

  const handleClearAll = () => {
    updateBusinessBrief({
      company: '',
      industry: '',
      businessModel: '',
      productsServices: '',
      offeringsFull: '',
      market: '',
      differentiator: '',
      financialHeadline: ''
    });
  };

  // Auto-fill plan title when company name changes
  useEffect(() => {
    const companyName = businessBrief.company;
    
    // Only auto-fill if:
    // 1. Company name exists
    // 2. Plan title is empty OR hasn't been manually edited
    // 3. Title is still default or empty
    if (companyName && !planTitleManuallyEdited && (!title || title === "Untitled Business Plan")) {
      const currentYear = new Date().getFullYear();
      const autoTitle = `${companyName} Business Plan – ${currentYear}`;
      setPlanTitle(autoTitle, false); // false = not a manual edit
    }
  }, [businessBrief.company, planTitleManuallyEdited, title, setPlanTitle]);

  const getSummaryText = () => {
    const parts = [];
    if (businessBrief.company) parts.push(businessBrief.company);
    if (businessBrief.businessModel) parts.push(businessBrief.businessModel);
    if (businessBrief.market) parts.push(businessBrief.market);
    if (parts.length === 0) return 'Click to add busines contect to help AI generate personalized conent.';
    return parts.join(' • ');
  };

  return (
    <div className="mb-4" data-testid="business-brief-section">
      <h2 className="text-lg font-semibold text-gray-900 dark:text-gray-100 mb-2 flex items-center gap-2">
        <Briefcase className="w-5 h-5 text-gray-900 dark:text-gray-100" />
        Business Brief
      </h2>
      <Card className="sticky top-0 z-10 shadow-md" data-testid="business-brief-card">
        <CardHeader 
          className="pb-3 cursor-pointer hover:bg-muted/50 transition-colors"
          onClick={() => setIsExpanded(!isExpanded)}
        >
          <div className="flex items-center justify-between">
            <div className="flex-1">
              {!isExpanded && (
                <p className="text-base font-bold text-muted-foreground">{getSummaryText()}</p>
              )}
            </div>
            <div className="flex gap-2">
              <Button 
                variant="outline" 
                size="sm" 
                onClick={(e) => {
                  e.stopPropagation();
                  handleClearAll();
                }}
                data-testid="button-clear-brief"
              >
                Clear
              </Button>
              <Button 
                variant="ghost" 
                size="sm" 
                onClick={(e) => {
                  e.stopPropagation();
                  setIsExpanded(!isExpanded);
                }}
                data-testid="toggle-business-brief"
              >
                {isExpanded ? <ChevronUp className="w-4 h-4" /> : <ChevronDown className="w-4 h-4" />}
              </Button>
            </div>
          </div>
        </CardHeader>

      {isExpanded && (
        <CardContent className="space-y-4">
          <p className="text-sm text-muted-foreground mb-4">
            Uses your Business Brief to draft or refine sections. This context helps AI generate personalized content instead of generic templates.
          </p>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2 md:col-span-2">
              <Label htmlFor="planTitle" className="text-sm font-medium">
                Plan Title
              </Label>
              <Input
                id="planTitle"
                value={title || ''}
                onChange={(e) => handlePlanTitleChange(e.target.value)}
                placeholder="e.g., IBrandBiz Business Plan – 2025"
                data-testid="input-plan-title"
              />
              <p className="text-xs text-muted-foreground">
                Used in previews, exports, and default file names.
              </p>
            </div>

            <div className="space-y-2">
              <Label htmlFor="company" className="text-sm font-medium">
                Company Name
              </Label>
              <Input
                id="company"
                value={businessBrief.company || ''}
                onChange={(e) => handleFieldChange('company', e.target.value)}
                placeholder="e.g., IBrandBiz"
                data-testid="input-company"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="industry" className="text-sm font-medium">
                Industry
              </Label>
              <Input
                id="industry"
                value={businessBrief.industry || ''}
                onChange={(e) => handleFieldChange('industry', e.target.value)}
                placeholder="e.g., Brand & Web Services"
                data-testid="input-industry"
              />
            </div>

            <div className="space-y-2 md:col-span-2">
              <Label htmlFor="businessModel" className="text-sm font-medium">
                Business Model
              </Label>
              <Input
                id="businessModel"
                value={businessBrief.businessModel || ''}
                onChange={(e) => handleFieldChange('businessModel', e.target.value)}
                placeholder="e.g., Global SaaS + Web Services"
                data-testid="input-business-model"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="productsServices" className="text-sm font-medium">
                Products / Services <span className="text-xs text-muted-foreground">(short bullets)</span>
              </Label>
              <Input
                id="productsServices"
                value={businessBrief.productsServices || ''}
                onChange={(e) => handleFieldChange('productsServices', e.target.value)}
                placeholder="e.g., Brand tools; Business Plan Template; Domains & Hosting"
                data-testid="input-products-services"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="market" className="text-sm font-medium">
                Target Market
              </Label>
              <Input
                id="market"
                value={businessBrief.market || ''}
                onChange={(e) => handleFieldChange('market', e.target.value)}
                placeholder="e.g., Entrepreneurs & SMEs (global)"
                data-testid="input-market"
              />
            </div>

            <div className="space-y-2 md:col-span-2">
              <Label htmlFor="offeringsFull" className="text-sm font-medium">
                Offerings (Full Suite) <span className="text-xs text-muted-foreground">(detailed list)</span>
              </Label>
              <Input
                id="offeringsFull"
                value={businessBrief.offeringsFull || ''}
                onChange={(e) => handleFieldChange('offeringsFull', e.target.value)}
                placeholder="e.g., Brand Name Wizard, Slogan Generator, Logo Creator, Brand Kit, Business Plan Developer; Business Plan Template; Domains & Web Hosting"
                data-testid="input-offerings-full"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="differentiator" className="text-sm font-medium">
                Competitive Edge
              </Label>
              <Input
                id="differentiator"
                value={businessBrief.differentiator || ''}
                onChange={(e) => handleFieldChange('differentiator', e.target.value)}
                placeholder="e.g., Integrated startup toolkit + hosting; partner marketing"
                data-testid="input-differentiator"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="financialHeadline" className="text-sm font-medium">
                Financial Headline <span className="text-muted-foreground">(Optional)</span>
              </Label>
              <Input
                id="financialHeadline"
                value={businessBrief.financialHeadline || ''}
                onChange={(e) => handleFieldChange('financialHeadline', e.target.value)}
                placeholder="e.g., Launch Q4 2025; 2026 $730k; 2027 $1.46M"
                data-testid="input-financial-headline"
              />
            </div>
          </div>
        </CardContent>
      )}
      </Card>
    </div>
  );
}
