import { useState, useEffect, useMemo, useCallback } from 'react';
import { Card } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { RefreshCw, Maximize2, Download, ExternalLink, Loader2, Monitor, Tablet, Smartphone } from 'lucide-react';
import { TemplateRenderer } from '@/services/template-renderer';
import type { WebsiteTemplate, TemplateCustomization } from '@shared/schema';
import DOMPurify from 'dompurify';

type DeviceMode = 'desktop' | 'tablet' | 'mobile';

interface TemplatePreviewProps {
  template: WebsiteTemplate;
  customization: TemplateCustomization;
  deviceMode: DeviceMode;
  zoomLevel: number;
  onFullscreen?: () => void;
  onExport?: () => void;
  onDeviceModeChange?: (mode: DeviceMode) => void;
  onZoomChange?: (zoom: number) => void;
}

export default function TemplatePreview({ 
  template, 
  customization, 
  deviceMode, 
  zoomLevel,
  onFullscreen,
  onExport,
  onDeviceModeChange,
  onZoomChange
}: TemplatePreviewProps) {
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [isFullscreen, setIsFullscreen] = useState(false);
  const [previewKey, setPreviewKey] = useState(0);
  const [isUpdating, setIsUpdating] = useState(false);
  const [previewMode, setPreviewMode] = useState<'live' | 'rendered'>('live');

  // Template renderer instance with memoization for performance
  const templateRenderer = useMemo(() => {
    try {
      return TemplateRenderer.create(template, customization);
    } catch (error) {
      console.error('Failed to create template renderer:', error);
      return null;
    }
  }, [template, customization]);

  // CSS variables for dynamic theming with memoization for performance
  const cssVariables = useMemo(() => {
    try {
      return templateRenderer?.generateStyles().cssVariables || {};
    } catch (error) {
      console.error('Failed to generate CSS variables:', error);
      return {};
    }
  }, [templateRenderer]);

  // Rendered HTML content
  const renderedHTML = useMemo(() => {
    try {
      if (!templateRenderer) return '<html><body><p>Template renderer not available</p></body></html>';
      return templateRenderer.generateHTML({
        deviceMode,
        includeInteractivity: false,
        optimizeForExport: false
      });
    } catch (error) {
      console.error('Failed to generate rendered HTML:', error);
      return '<html><body><p>Error generating preview</p></body></html>';
    }
  }, [templateRenderer, deviceMode]);

  const getSpacingValue = useCallback((spacing: string) => {
    switch (spacing) {
      case 'compact': return '0.5rem';
      case 'relaxed': return '2rem';
      case 'loose': return '3rem';
      default: return '1.5rem';
    }
  }, []);

  const getDeviceStyles = useCallback(() => {
    const baseStyles = {
      transition: 'all 0.3s cubic-bezier(0.4, 0, 0.2, 1)',
      transformOrigin: 'top left',
      transform: `scale(${zoomLevel / 100})`,
      ...cssVariables,
    };

    switch (deviceMode) {
      case 'mobile':
        return {
          ...baseStyles,
          width: '375px',
          height: '812px',
          maxWidth: '375px',
        };
      case 'tablet':
        return {
          ...baseStyles,
          width: '768px',
          height: '1024px',
          maxWidth: '768px',
        };
      default: // desktop
        return {
          ...baseStyles,
          width: '100%',
          height: '100%',
          maxWidth: 'var(--layout-max-width)',
        };
    }
  }, [deviceMode, zoomLevel, cssVariables]);

  // Trigger update animation when customization changes
  useEffect(() => {
    setIsUpdating(true);
    const timer = setTimeout(() => setIsUpdating(false), 200);
    return () => clearTimeout(timer);
  }, [customization]);

  const handleRefresh = useCallback(() => {
    setIsRefreshing(true);
    setPreviewKey(prev => prev + 1);
    setTimeout(() => setIsRefreshing(false), 1000);
  }, []);

  const handleFullscreen = useCallback(() => {
    setIsFullscreen(!isFullscreen);
    onFullscreen?.();
  }, [isFullscreen, onFullscreen]);

  // Removed duplicate HTML generation code - using TemplateRenderer.generateHTML() instead

  const handleExport = useCallback(() => {
    try {
      if (!templateRenderer) {
        console.error('Template renderer not available for export');
        return;
      }
      const htmlContent = templateRenderer.generateHTML({
        deviceMode: 'desktop',
        includeInteractivity: true,
        optimizeForExport: true
      });
      const blob = new Blob([htmlContent], { type: 'text/html' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `${template.name?.replace(/[^a-zA-Z0-9]/g, '-') || 'template'}-preview.html`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
      onExport?.();
    } catch (error) {
      console.error('Export failed:', error);
      // Could show a toast notification here
    }
  }, [templateRenderer, template.name, onExport]);

  const handleDeviceModeChange = useCallback((mode: DeviceMode) => {
    onDeviceModeChange?.(mode);
  }, [onDeviceModeChange]);

  const handleZoomChange = useCallback((zoom: number) => {
    onZoomChange?.(zoom);
  }, [onZoomChange]);

  const handlePreviewInNewTab = useCallback(() => {
    try {
      if (!templateRenderer) {
        console.error('Template renderer not available for preview');
        return;
      }
      const htmlContent = templateRenderer.generateHTML({
        deviceMode: 'desktop',
        includeInteractivity: true,
        optimizeForExport: false
      });
      const blob = new Blob([htmlContent], { type: 'text/html' });
      const url = URL.createObjectURL(blob);
      window.open(url, '_blank');
      // Clean up after a delay to allow the new tab to load
      setTimeout(() => URL.revokeObjectURL(url), 1000);
    } catch (error) {
      console.error('Preview in new tab failed:', error);
    }
  }, [templateRenderer]);

  const renderRealisticContent = useCallback(() => {
    if (customization.content.sections.length === 0) {
      return (
        <>
          {/* Hero Section */}
          <section className="template-section hero-section" data-section-type="hero">
            <div className="section-content">
              <h1 className="hero-title">{DOMPurify.sanitize(template.name || 'Your Amazing Website', { ALLOWED_TAGS: [], KEEP_CONTENT: true })}</h1>
              <p className="hero-subtitle">
                {DOMPurify.sanitize(template.description || 'Create something extraordinary with our customizable templates', { ALLOWED_TAGS: ['br'], KEEP_CONTENT: true })}
              </p>
              <div className="hero-actions">
                <a href="#" className="btn btn-primary">Get Started</a>
                <a href="#" className="btn btn-secondary">View Demo</a>
              </div>
            </div>
          </section>

          {/* Features Section */}
          <section className="template-section features-section" data-section-type="features">
            <div className="section-content">
              <h2 className="section-title">Why Choose Us</h2>
              <div className="features-grid">
                <div className="feature-item">
                  <h3>Fast & Reliable</h3>
                  <p>Built with performance in mind for optimal user experience</p>
                </div>
                <div className="feature-item">
                  <h3>Customizable</h3>
                  <p>Tailor every aspect to match your brand and vision</p>
                </div>
                <div className="feature-item">
                  <h3>Professional</h3>
                  <p>Modern design that looks great on all devices</p>
                </div>
              </div>
            </div>
          </section>

          {/* About Section */}
          <section className="template-section about-section" data-section-type="about">
            <div className="section-content">
              <h2 className="section-title">About Our Company</h2>
              <p className="section-text">
                We are dedicated to providing exceptional service and innovative solutions that help your business grow and succeed in today's competitive market.
              </p>
            </div>
          </section>
        </>
      );
    }

    return customization.content.sections
      .sort((a, b) => a.order - b.order)
      .map((section, index) => {
        const isEven = index % 2 === 0;
        return (
          <section 
            key={section.id} 
            className={`template-section ${section.type}-section ${isEven ? 'even' : 'odd'}`}
            data-section-type={section.type}
          >
            <div className="section-content">
              <h2 className="section-title">
                {DOMPurify.sanitize(section.content?.title || `${section.type} Section`, { ALLOWED_TAGS: [], KEEP_CONTENT: true })}
              </h2>
              <div className="section-text">
                {DOMPurify.sanitize(section.content?.text || `This is a ${section.type} section with customized styling.`, { ALLOWED_TAGS: ['br'], KEEP_CONTENT: true })}
              </div>
              {section.type === 'hero' && (
                <div className="hero-actions">
                  <a href="#" className="btn btn-primary">Get Started</a>
                  <a href="#" className="btn btn-secondary">Learn More</a>
                </div>
              )}
              {section.type === 'contact' && (
                <div className="contact-form">
                  <div className="form-group">
                    <input type="text" placeholder="Your Name" className="form-input" />
                  </div>
                  <div className="form-group">
                    <input type="email" placeholder="Your Email" className="form-input" />
                  </div>
                  <div className="form-group">
                    <textarea placeholder="Your Message" className="form-textarea"></textarea>
                  </div>
                  <button className="btn btn-primary">Send Message</button>
                </div>
              )}
            </div>
          </section>
        );
      });
  }, [customization.content.sections, template]);

  const previewStyles = `
    .preview-container {
      background-color: var(--background);
      color: var(--text);
      font-family: var(--font-body);
      line-height: 1.6;
      min-height: 100%;
      border-radius: 8px;
      overflow: hidden;
      box-shadow: 0 8px 32px rgba(0,0,0,0.1);
      transition: all 0.3s ease;
    }
    
    .template-section {
      padding: 3rem 0;
      transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    }
    
    .template-section.even {
      background-color: color-mix(in srgb, var(--primary) 5%, transparent);
    }
    
    .template-section.hero-section {
      background: linear-gradient(135deg, var(--primary), var(--secondary));
      color: white;
      text-align: center;
      padding: 6rem 0;
    }
    
    .section-content {
      max-width: var(--layout-max-width);
      margin: 0 auto;
      padding: 0 var(--layout-spacing);
    }
    
    .section-title, .hero-title {
      font-family: var(--font-heading);
      font-weight: 700;
      margin-bottom: 1rem;
      line-height: 1.2;
    }
    
    .hero-title {
      font-size: var(--font-size-h1);
      margin-bottom: 1rem;
    }
    
    .section-title {
      font-size: var(--font-size-h2);
      color: var(--primary);
      text-align: center;
      margin-bottom: 2rem;
    }
    
    .hero-section .section-title,
    .hero-section .hero-title {
      color: white;
    }
    
    .section-text, .hero-subtitle {
      font-size: var(--font-size-body);
      margin-bottom: 1.5rem;
      opacity: 0.9;
      text-align: center;
    }
    
    .hero-actions {
      display: flex;
      gap: 1rem;
      justify-content: center;
      flex-wrap: wrap;
      margin-top: 2rem;
    }
    
    .btn {
      display: inline-block;
      padding: 0.75rem 1.5rem;
      text-decoration: none;
      border-radius: 0.5rem;
      font-weight: 600;
      transition: all 0.3s ease;
      border: none;
      cursor: pointer;
      text-align: center;
    }
    
    .btn-primary {
      background-color: var(--accent);
      color: white;
    }
    
    .btn:hover {
      transform: translateY(-2px);
      box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    }
    
    .btn-secondary {
      background-color: transparent;
      color: white;
      border: 2px solid white;
    }
    
    .btn-secondary:hover {
      background-color: white;
      color: var(--primary);
    }
    
    .features-grid {
      display: grid;
      grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
      gap: 2rem;
      margin-top: 2rem;
    }
    
    .feature-item {
      text-align: center;
      padding: 2rem;
      border-radius: 0.5rem;
      background-color: var(--background);
      box-shadow: 0 2px 8px rgba(0,0,0,0.1);
      transition: transform 0.3s ease;
    }
    
    .feature-item:hover {
      transform: translateY(-4px);
    }
    
    .feature-item h3 {
      font-family: var(--font-heading);
      font-size: var(--font-size-h3);
      color: var(--primary);
      margin-bottom: 0.5rem;
    }
    
    .contact-form {
      max-width: 500px;
      margin: 2rem auto 0;
    }
    
    .form-group {
      margin-bottom: 1rem;
    }
    
    .form-input, .form-textarea {
      width: 100%;
      padding: 0.75rem;
      border: 2px solid color-mix(in srgb, var(--primary) 20%, transparent);
      border-radius: 0.5rem;
      font-size: var(--font-size-body);
      transition: border-color 0.3s ease;
    }
    
    .form-input:focus, .form-textarea:focus {
      outline: none;
      border-color: var(--primary);
    }
    
    .form-textarea {
      min-height: 100px;
      resize: vertical;
    }
    
    .footer-section {
      background-color: var(--text);
      color: var(--background);
      text-align: center;
      padding: 2rem 0;
    }
    
    .updating {
      opacity: 0.7;
      transform: scale(0.99);
    }
    
    @media (max-width: 768px) {
      .hero-actions {
        flex-direction: column;
        align-items: center;
      }
      
      .features-grid {
        grid-template-columns: 1fr;
      }
      
      .template-section {
        padding: 2rem 0;
      }
      
      .section-content {
        padding: 0 1rem;
      }
    }
  `;

  return (
    <div className="h-full flex flex-col">
      {/* Preview Header */}
      <div className="space-y-3 mb-4">
        {/* Top Row - Template Info & Controls */}
        <div className="flex items-center justify-between p-3 bg-muted/50 rounded-lg">
          <div className="flex items-center space-x-2">
            <Badge variant="outline">{template.name}</Badge>
            <Badge variant="secondary" className="capitalize">{deviceMode}</Badge>
            <Badge variant="outline">{zoomLevel}%</Badge>
            {isUpdating && (
              <Badge variant="default" className="animate-pulse">
                <Loader2 className="w-3 h-3 mr-1 animate-spin" />
                Updating...
              </Badge>
            )}
            {isRefreshing && (
              <Badge variant="default" className="animate-pulse">
                Refreshing...
              </Badge>
            )}
          </div>
          <div className="flex items-center space-x-2">
            <Button
              variant="ghost"
              size="sm"
              onClick={handleRefresh}
              disabled={isRefreshing}
              data-testid="button-refresh-preview"
            >
              <RefreshCw className={`w-4 h-4 ${isRefreshing ? 'animate-spin' : ''}`} />
            </Button>
            <Button
              variant="ghost"
              size="sm"
              onClick={handlePreviewInNewTab}
              data-testid="button-preview-new-tab"
            >
              <ExternalLink className="w-4 h-4" />
            </Button>
            <Button
              variant="ghost"
              size="sm"
              onClick={handleFullscreen}
              data-testid="button-fullscreen-preview"
            >
              <Maximize2 className="w-4 h-4" />
            </Button>
            <Button
              variant="ghost"
              size="sm"
              onClick={handleExport}
              data-testid="button-export-preview"
            >
              <Download className="w-4 h-4" />
            </Button>
          </div>
        </div>

        {/* Bottom Row - Device & Zoom Controls */}
        <div className="flex items-center justify-between p-3 bg-muted/30 rounded-lg">
          {/* Device Mode Controls */}
          <div className="flex items-center space-x-1">
            <span className="text-sm font-medium mr-2">Device:</span>
            <Button
              variant={deviceMode === 'desktop' ? 'default' : 'ghost'}
              size="sm"
              onClick={() => handleDeviceModeChange('desktop')}
              data-testid="button-desktop-mode"
            >
              <Monitor className="w-4 h-4" />
            </Button>
            <Button
              variant={deviceMode === 'tablet' ? 'default' : 'ghost'}
              size="sm"
              onClick={() => handleDeviceModeChange('tablet')}
              data-testid="button-tablet-mode"
            >
              <Tablet className="w-4 h-4" />
            </Button>
            <Button
              variant={deviceMode === 'mobile' ? 'default' : 'ghost'}
              size="sm"
              onClick={() => handleDeviceModeChange('mobile')}
              data-testid="button-mobile-mode"
            >
              <Smartphone className="w-4 h-4" />
            </Button>
          </div>

          {/* Zoom Controls */}
          <div className="flex items-center space-x-2">
            <span className="text-sm font-medium">Zoom:</span>
            <div className="flex items-center space-x-1">
              {[50, 75, 100, 125].map((zoom) => (
                <Button
                  key={zoom}
                  variant={zoomLevel === zoom ? 'default' : 'ghost'}
                  size="sm"
                  onClick={() => handleZoomChange(zoom)}
                  data-testid={`button-zoom-${zoom}`}
                >
                  {zoom}%
                </Button>
              ))}
            </div>
          </div>

          {/* Preview Mode Toggle */}
          <div className="flex items-center space-x-2">
            <span className="text-sm font-medium">Mode:</span>
            <div className="flex items-center space-x-1">
              <Button
                variant={previewMode === 'live' ? 'default' : 'ghost'}
                size="sm"
                onClick={() => setPreviewMode('live')}
                data-testid="button-preview-live"
              >
                Live
              </Button>
              <Button
                variant={previewMode === 'rendered' ? 'default' : 'ghost'}
                size="sm"
                onClick={() => setPreviewMode('rendered')}
                data-testid="button-preview-rendered"
              >
                Rendered
              </Button>
            </div>
          </div>
        </div>
      </div>

      {/* Preview Container */}
      <div className="flex-1 overflow-auto bg-muted/30 rounded-lg p-4">
        <div className="mx-auto" style={getDeviceStyles()}>
          {previewMode === 'live' ? (
            <div 
              className={`preview-container ${isUpdating ? 'updating' : ''}`} 
              key={previewKey}
            >
              <style>{previewStyles}</style>
              
              {renderRealisticContent()}
              
              {/* Footer */}
              <section className="template-section footer-section">
                <div className="section-content">
                  <p style={{ fontSize: 'var(--font-size-body)', margin: 0 }}>
                    © 2025 {DOMPurify.sanitize(template.name || 'Your Website', { ALLOWED_TAGS: [], KEEP_CONTENT: true })}. All rights reserved.
                  </p>
                </div>
              </section>
            </div>
          ) : (
            <div 
              className={`preview-container rendered-preview ${isUpdating ? 'updating' : ''}`}
              key={`rendered-${previewKey}`}
            >
              <iframe
                srcDoc={renderedHTML}
                style={{
                  width: '100%',
                  height: '100%',
                  border: 'none',
                  borderRadius: '8px',
                  backgroundColor: 'white'
                }}
                title="Rendered Preview"
                data-testid="rendered-preview-iframe"
                sandbox="allow-same-origin allow-scripts"
                referrerPolicy="no-referrer"
              />
            </div>
          )}
        </div>
      </div>

      {/* Preview Info */}
      <div className="mt-4 p-3 bg-muted/50 rounded-lg">
        <div className="flex items-center justify-between text-sm text-muted-foreground">
          <span>
            {previewMode === 'live' ? 'Live Preview' : 'Rendered Preview'} | 
            {customization.content.sections.length} sections
          </span>
          <span>
            Template: {template.templateType || 'Custom'} | 
            Last updated: {new Date().toLocaleTimeString()}
          </span>
        </div>
      </div>
    </div>
  );
}