import { useState, useMemo } from 'react';
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from '@/components/ui/alert-dialog';
import { Badge } from '@/components/ui/badge';
import { Separator } from '@/components/ui/separator';
import { ScrollArea } from '@/components/ui/scroll-area';
import {
  Search,
  MoreVertical,
  Copy,
  Trash2,
  Download,
  Eye,
  Heart,
  Star,
  Calendar,
  Filter,
  Loader2,
  FolderOpen,
  Grid3X3,
  List,
} from 'lucide-react';
import { toast } from '@/hooks/use-toast';
import { apiRequest } from '@/lib/queryClient';
import { format } from 'date-fns';
import type { UserTemplateCustomization, WebsiteTemplate } from '@shared/schema';

interface TemplateLibraryProps {
  onLoadTemplate?: (customization: UserTemplateCustomization) => void;
  onDuplicateTemplate?: (customization: UserTemplateCustomization) => void;
  isOpen?: boolean;
  onOpenChange?: (open: boolean) => void;
  children?: React.ReactNode;
}

type ViewMode = 'grid' | 'list';
type SortBy = 'recent' | 'name' | 'favorite';
type FilterBy = 'all' | 'favorites' | 'templates';

export default function TemplateLibrary({
  onLoadTemplate,
  onDuplicateTemplate,
  isOpen,
  onOpenChange,
  children
}: TemplateLibraryProps) {
  const [open, setOpen] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');
  const [sortBy, setSortBy] = useState<SortBy>('recent');
  const [filterBy, setFilterBy] = useState<FilterBy>('all');
  const [viewMode, setViewMode] = useState<ViewMode>('grid');
  const [deleteConfirmOpen, setDeleteConfirmOpen] = useState(false);
  const [templateToDelete, setTemplateToDelete] = useState<UserTemplateCustomization | null>(null);
  
  const queryClient = useQueryClient();
  
  const isControlled = isOpen !== undefined && onOpenChange !== undefined;
  const dialogOpen = isControlled ? isOpen : open;
  const setDialogOpen = isControlled ? onOpenChange : setOpen;

  // Fetch user's template customizations
  const { data: customizations = [], isLoading } = useQuery({
    queryKey: ['/api/template-customizations'],
    enabled: dialogOpen,
  });

  // Delete template mutation
  const deleteTemplateMutation = useMutation({
    mutationFn: async (id: string) => {
      return apiRequest('DELETE', `/api/template-customizations/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/template-customizations'] });
      toast({
        title: "Template Deleted",
        description: "Template has been permanently deleted.",
      });
      setDeleteConfirmOpen(false);
      setTemplateToDelete(null);
    },
    onError: (error: any) => {
      toast({
        title: "Delete Failed",
        description: error.message || "Failed to delete template. Please try again.",
        variant: "destructive",
      });
    },
  });

  // Duplicate template mutation
  const duplicateTemplateMutation = useMutation({
    mutationFn: async ({ id, name }: { id: string; name: string }) => {
      return apiRequest('POST', `/api/template-customizations/${id}/duplicate`, { name });
    },
    onSuccess: (duplicatedTemplate: UserTemplateCustomization) => {
      queryClient.invalidateQueries({ queryKey: ['/api/template-customizations'] });
      toast({
        title: "Template Duplicated",
        description: `Template "${duplicatedTemplate.customizationName}" has been created.`,
      });
      onDuplicateTemplate?.(duplicatedTemplate);
    },
    onError: (error: any) => {
      toast({
        title: "Duplication Failed",
        description: error.message || "Failed to duplicate template. Please try again.",
        variant: "destructive",
      });
    },
  });

  // Toggle favorite mutation
  const toggleFavoriteMutation = useMutation({
    mutationFn: async ({ id, isFavorite }: { id: string; isFavorite: boolean }) => {
      return apiRequest('PUT', `/api/template-customizations/${id}`, { isFavorite });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/template-customizations'] });
    },
    onError: (error: any) => {
      toast({
        title: "Update Failed",
        description: error.message || "Failed to update template.",
        variant: "destructive",
      });
    },
  });

  // Filter and sort templates
  const filteredAndSortedTemplates = useMemo(() => {
    const templateList = Array.isArray(customizations) ? customizations as UserTemplateCustomization[] : [];
    let filtered = templateList.filter((template: UserTemplateCustomization) => {
      // Search filter
      const configurations = template.configurations as any || {};
      const description = configurations.metadata?.description || '';
      const matchesSearch = template.customizationName?.toLowerCase().includes(searchQuery.toLowerCase()) ||
                           description.toLowerCase().includes(searchQuery.toLowerCase());
      
      // Category filter
      let matchesFilter = true;
      if (filterBy === 'favorites') {
        matchesFilter = template.isFavorite === true;
      } else if (filterBy === 'templates') {
        matchesFilter = template.isTemplate === true;
      }
      
      return matchesSearch && matchesFilter;
    });

    // Sort templates
    filtered.sort((a: UserTemplateCustomization, b: UserTemplateCustomization) => {
      switch (sortBy) {
        case 'name':
          return (a.customizationName || '').localeCompare(b.customizationName || '');
        case 'favorite':
          if (a.isFavorite && !b.isFavorite) return -1;
          if (!a.isFavorite && b.isFavorite) return 1;
          return new Date(b.updatedAt || b.createdAt || 0).getTime() - new Date(a.updatedAt || a.createdAt || 0).getTime();
        case 'recent':
        default:
          return new Date(b.updatedAt || b.createdAt || 0).getTime() - new Date(a.updatedAt || a.createdAt || 0).getTime();
      }
    });

    return filtered;
  }, [customizations, searchQuery, sortBy, filterBy]);

  const handleLoadTemplate = (template: UserTemplateCustomization) => {
    onLoadTemplate?.(template);
    setDialogOpen(false);
    toast({
      title: "Template Loaded",
      description: `"${template.customizationName}" has been loaded successfully.`,
    });
  };

  const handleDuplicateTemplate = async (template: UserTemplateCustomization) => {
    const newName = `${template.customizationName} (Copy)`;
    duplicateTemplateMutation.mutate({ id: template.id, name: newName });
  };

  const handleDeleteTemplate = (template: UserTemplateCustomization) => {
    setTemplateToDelete(template);
    setDeleteConfirmOpen(true);
  };

  const confirmDelete = () => {
    if (templateToDelete) {
      deleteTemplateMutation.mutate(templateToDelete.id);
    }
  };

  const handleToggleFavorite = (template: UserTemplateCustomization) => {
    toggleFavoriteMutation.mutate({
      id: template.id,
      isFavorite: !template.isFavorite
    });
  };

  const handleExportTemplate = async (template: UserTemplateCustomization) => {
    try {
      const data = {
        name: template.customizationName,
        customization: template.configurations,
        contentOverrides: template.contentOverrides,
        styleOverrides: template.styleOverrides,
        exportedAt: new Date().toISOString(),
      };
      
      const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `${template.customizationName?.replace(/[^a-z0-9]/gi, '_') || 'template'}.json`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
      
      toast({
        title: "Template Exported",
        description: "Template has been exported as JSON file.",
      });
    } catch (error) {
      toast({
        title: "Export Failed",
        description: "Failed to export template. Please try again.",
        variant: "destructive",
      });
    }
  };

  const getPreviewColors = (template: UserTemplateCustomization) => {
    const styleOverrides = template.styleOverrides as any || {};
    const configurations = template.configurations as any || {};
    const colors = styleOverrides.colors || configurations.colors || {};
    return {
      primary: colors.primary || '#00C851',
      secondary: colors.secondary || '#427AB9',
      accent: colors.accent || '#FF6F00',
    };
  };

  const TemplateCard = ({ template }: { template: UserTemplateCustomization }) => {
    const colors = getPreviewColors(template);
    const configurations = template.configurations as any || {};
    const description = configurations.metadata?.description;
    const lastModified = format(new Date(template.updatedAt || template.createdAt || 0), 'MMM dd, yyyy');

    if (viewMode === 'list') {
      return (
        <div className="flex items-center space-x-4 p-4 border rounded-lg hover:bg-accent/50 transition-colors">
          <div 
            className="w-16 h-12 rounded border flex-shrink-0"
            style={{ 
              background: `linear-gradient(135deg, ${colors.primary}, ${colors.secondary})` 
            }}
          />
          <div className="flex-1 min-w-0">
            <div className="flex items-center gap-2 mb-1">
              <h3 className="font-medium truncate" data-testid={`template-name-${template.id}`}>
                {template.customizationName || 'Untitled Template'}
              </h3>
              {template.isFavorite && <Heart className="w-4 h-4 fill-red-500 text-red-500" />}
              {template.isTemplate && <Star className="w-4 h-4 fill-yellow-500 text-yellow-500" />}
            </div>
            {description && (
              <p className="text-sm text-muted-foreground truncate">{description}</p>
            )}
            <p className="text-xs text-muted-foreground flex items-center gap-1 mt-1">
              <Calendar className="w-3 h-3" />
              {lastModified}
            </p>
          </div>
          <div className="flex items-center space-x-2">
            <Button
              size="sm"
              onClick={() => handleLoadTemplate(template)}
              data-testid={`button-load-${template.id}`}
            >
              Load
            </Button>
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="ghost" size="sm" data-testid={`button-menu-${template.id}`}>
                  <MoreVertical className="w-4 h-4" />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="end">
                <DropdownMenuItem onClick={() => handleToggleFavorite(template)}>
                  <Heart className="w-4 h-4 mr-2" />
                  {template.isFavorite ? 'Remove from Favorites' : 'Add to Favorites'}
                </DropdownMenuItem>
                <DropdownMenuItem onClick={() => handleDuplicateTemplate(template)}>
                  <Copy className="w-4 h-4 mr-2" />
                  Duplicate
                </DropdownMenuItem>
                <DropdownMenuItem onClick={() => handleExportTemplate(template)}>
                  <Download className="w-4 h-4 mr-2" />
                  Export JSON
                </DropdownMenuItem>
                <DropdownMenuSeparator />
                <DropdownMenuItem 
                  onClick={() => handleDeleteTemplate(template)}
                  className="text-destructive"
                >
                  <Trash2 className="w-4 h-4 mr-2" />
                  Delete
                </DropdownMenuItem>
              </DropdownMenuContent>
            </DropdownMenu>
          </div>
        </div>
      );
    }

    return (
      <Card className="group hover:shadow-lg transition-all duration-200 cursor-pointer">
        <CardHeader className="relative p-0">
          <div 
            className="h-32 rounded-t-lg relative overflow-hidden"
            style={{ 
              background: `linear-gradient(135deg, ${colors.primary}, ${colors.secondary})` 
            }}
          >
            <div className="absolute inset-4 bg-white/90 rounded shadow-sm">
              <div className="h-3 bg-gray-200 rounded w-3/4 m-2"></div>
              <div className="h-2 bg-gray-100 rounded w-1/2 m-2"></div>
              <div className="grid grid-cols-3 gap-1 m-2">
                <div className="h-8 bg-gray-50 rounded"></div>
                <div className="h-8 bg-gray-50 rounded"></div>
                <div className="h-8 bg-gray-50 rounded"></div>
              </div>
            </div>
            <div className="absolute top-2 right-2 flex gap-1">
              {template.isFavorite && (
                <Badge variant="secondary" className="text-xs">
                  <Heart className="w-3 h-3 fill-red-500 text-red-500" />
                </Badge>
              )}
              {template.isTemplate && (
                <Badge variant="secondary" className="text-xs">
                  <Star className="w-3 h-3 fill-yellow-500 text-yellow-500" />
                </Badge>
              )}
            </div>
          </div>
        </CardHeader>
        <CardContent className="p-4">
          <div className="flex items-start justify-between mb-2">
            <CardTitle className="text-sm font-medium truncate" data-testid={`template-name-${template.id}`}>
              {template.customizationName || 'Untitled Template'}
            </CardTitle>
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button 
                  variant="ghost" 
                  size="sm" 
                  className="opacity-0 group-hover:opacity-100 transition-opacity"
                  data-testid={`button-menu-${template.id}`}
                >
                  <MoreVertical className="w-4 h-4" />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="end">
                <DropdownMenuItem onClick={() => handleToggleFavorite(template)}>
                  <Heart className="w-4 h-4 mr-2" />
                  {template.isFavorite ? 'Remove from Favorites' : 'Add to Favorites'}
                </DropdownMenuItem>
                <DropdownMenuItem onClick={() => handleDuplicateTemplate(template)}>
                  <Copy className="w-4 h-4 mr-2" />
                  Duplicate
                </DropdownMenuItem>
                <DropdownMenuItem onClick={() => handleExportTemplate(template)}>
                  <Download className="w-4 h-4 mr-2" />
                  Export JSON
                </DropdownMenuItem>
                <DropdownMenuSeparator />
                <DropdownMenuItem 
                  onClick={() => handleDeleteTemplate(template)}
                  className="text-destructive"
                >
                  <Trash2 className="w-4 h-4 mr-2" />
                  Delete
                </DropdownMenuItem>
              </DropdownMenuContent>
            </DropdownMenu>
          </div>
          
          {description && (
            <p className="text-xs text-muted-foreground mb-3 line-clamp-2">{description}</p>
          )}
          
          <div className="flex items-center justify-between">
            <p className="text-xs text-muted-foreground flex items-center gap-1">
              <Calendar className="w-3 h-3" />
              {lastModified}
            </p>
            <Button
              size="sm"
              onClick={() => handleLoadTemplate(template)}
              data-testid={`button-load-${template.id}`}
            >
              Load
            </Button>
          </div>
        </CardContent>
      </Card>
    );
  };

  return (
    <>
      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        {children && (
          <DialogTrigger asChild>
            {children}
          </DialogTrigger>
        )}
        <DialogContent className="max-w-4xl max-h-[80vh] flex flex-col" data-testid="dialog-template-library">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <FolderOpen className="w-5 h-5" />
              Template Library
            </DialogTitle>
            <DialogDescription>
              Browse, load, and manage your saved template customizations.
            </DialogDescription>
          </DialogHeader>

          {/* Search and Filters */}
          <div className="flex flex-col sm:flex-row gap-4 py-4">
            <div className="relative flex-1">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
              <Input
                placeholder="Search templates..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pl-10"
                data-testid="input-search-templates"
              />
            </div>
            
            <div className="flex gap-2">
              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button variant="outline" size="sm" data-testid="button-filter">
                    <Filter className="w-4 h-4 mr-2" />
                    Filter
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent>
                  <DropdownMenuItem onClick={() => setFilterBy('all')}>
                    All Templates
                  </DropdownMenuItem>
                  <DropdownMenuItem onClick={() => setFilterBy('favorites')}>
                    Favorites Only
                  </DropdownMenuItem>
                  <DropdownMenuItem onClick={() => setFilterBy('templates')}>
                    Shared Templates
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>

              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button variant="outline" size="sm" data-testid="button-sort">
                    Sort: {sortBy === 'recent' ? 'Recent' : sortBy === 'name' ? 'Name' : 'Favorites'}
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent>
                  <DropdownMenuItem onClick={() => setSortBy('recent')}>
                    Most Recent
                  </DropdownMenuItem>
                  <DropdownMenuItem onClick={() => setSortBy('name')}>
                    Name (A-Z)
                  </DropdownMenuItem>
                  <DropdownMenuItem onClick={() => setSortBy('favorite')}>
                    Favorites First
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>

              <div className="flex rounded-md border">
                <Button
                  variant={viewMode === 'grid' ? 'default' : 'ghost'}
                  size="sm"
                  onClick={() => setViewMode('grid')}
                  className="rounded-r-none"
                  data-testid="button-view-grid"
                >
                  <Grid3X3 className="w-4 h-4" />
                </Button>
                <Button
                  variant={viewMode === 'list' ? 'default' : 'ghost'}
                  size="sm"
                  onClick={() => setViewMode('list')}
                  className="rounded-l-none border-l"
                  data-testid="button-view-list"
                >
                  <List className="w-4 h-4" />
                </Button>
              </div>
            </div>
          </div>

          <Separator />

          {/* Templates Grid/List */}
          <ScrollArea className="flex-1">
            {isLoading ? (
              <div className="flex items-center justify-center py-12">
                <Loader2 className="w-8 h-8 animate-spin" />
                <span className="ml-2">Loading templates...</span>
              </div>
            ) : filteredAndSortedTemplates.length > 0 ? (
              <div className={viewMode === 'grid' 
                ? "grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 p-1" 
                : "space-y-2 p-1"
              }>
                {filteredAndSortedTemplates.map((template: UserTemplateCustomization) => (
                  <TemplateCard key={template.id} template={template} />
                ))}
              </div>
            ) : (
              <div className="flex flex-col items-center justify-center py-12 text-center">
                <FolderOpen className="w-12 h-12 text-muted-foreground mb-4" />
                <h3 className="text-lg font-medium mb-2">No Templates Found</h3>
                <p className="text-muted-foreground">
                  {searchQuery ? 'No templates match your search criteria.' : 'You haven\'t saved any templates yet.'}
                </p>
                {searchQuery && (
                  <Button 
                    variant="outline" 
                    size="sm" 
                    onClick={() => setSearchQuery('')}
                    className="mt-3"
                  >
                    Clear Search
                  </Button>
                )}
              </div>
            )}
          </ScrollArea>
        </DialogContent>
      </Dialog>

      {/* Delete Confirmation Dialog */}
      <AlertDialog open={deleteConfirmOpen} onOpenChange={setDeleteConfirmOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Delete Template</AlertDialogTitle>
            <AlertDialogDescription>
              Are you sure you want to delete "{templateToDelete?.customizationName}"? 
              This action cannot be undone.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-delete">Cancel</AlertDialogCancel>
            <AlertDialogAction
              onClick={confirmDelete}
              disabled={deleteTemplateMutation.isPending}
              className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
              data-testid="button-confirm-delete"
            >
              {deleteTemplateMutation.isPending ? (
                <>
                  <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  Deleting...
                </>
              ) : (
                'Delete'
              )}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </>
  );
}