import { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { DragDropContext, Droppable, Draggable } from '@hello-pangea/dnd';
import { 
  GripVertical, 
  Plus, 
  Trash2, 
  Eye, 
  EyeOff,
  Settings,
  ChevronDown,
  ChevronRight
} from 'lucide-react';
import type { WebsiteTemplate } from '@shared/schema';

interface Section {
  id: string;
  type: string;
  content: any;
  order: number;
}

interface SectionManagerProps {
  sections: Section[];
  onChange: (sections: Section[]) => void;
  template: WebsiteTemplate;
}

interface SectionItem extends Section {
  isVisible: boolean;
  isExpanded: boolean;
}

const sectionTypes = [
  { value: 'header', label: 'Header', icon: '📄' },
  { value: 'hero', label: 'Hero Section', icon: '🦸' },
  { value: 'about', label: 'About', icon: 'ℹ️' },
  { value: 'services', label: 'Services', icon: '🛠️' },
  { value: 'gallery', label: 'Gallery', icon: '🖼️' },
  { value: 'contact', label: 'Contact', icon: '📞' },
  { value: 'footer', label: 'Footer', icon: '📝' }
];

export default function SectionManager({ sections, onChange, template }: SectionManagerProps) {
  const [sectionItems, setSectionItems] = useState<SectionItem[]>(
    sections.map(section => ({
      ...section,
      isVisible: true,
      isExpanded: false
    }))
  );

  const handleDragEnd = (result: any) => {
    if (!result.destination) return;

    const items = Array.from(sectionItems);
    const [reorderedItem] = items.splice(result.source.index, 1);
    items.splice(result.destination.index, 0, reorderedItem);

    // Update order numbers
    const updatedItems = items.map((item, index) => ({
      ...item,
      order: index
    }));

    setSectionItems(updatedItems);
    onChange(updatedItems);
  };

  const addSection = (type: string) => {
    const newSection: SectionItem = {
      id: `section-${Date.now()}`,
      type,
      content: { title: '', text: '' },
      order: sectionItems.length,
      isVisible: true,
      isExpanded: false
    };

    const updatedItems = [...sectionItems, newSection];
    setSectionItems(updatedItems);
    onChange(updatedItems);
  };

  const removeSection = (sectionId: string) => {
    const updatedItems = sectionItems.filter(item => item.id !== sectionId);
    setSectionItems(updatedItems);
    onChange(updatedItems);
  };

  const toggleVisibility = (sectionId: string) => {
    setSectionItems(prev => 
      prev.map(item => 
        item.id === sectionId 
          ? { ...item, isVisible: !item.isVisible }
          : item
      )
    );
  };

  const toggleExpanded = (sectionId: string) => {
    setSectionItems(prev => 
      prev.map(item => 
        item.id === sectionId 
          ? { ...item, isExpanded: !item.isExpanded }
          : item
      )
    );
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle>Section Manager</CardTitle>
      </CardHeader>
      <CardContent className="space-y-4">
        {/* Add Section Buttons */}
        <div className="border rounded-lg p-3">
          <h4 className="font-medium mb-3">Add Sections</h4>
          <div className="grid grid-cols-2 gap-2">
            {sectionTypes.map((type) => (
              <Button
                key={type.value}
                variant="outline"
                size="sm"
                onClick={() => addSection(type.value)}
                className="justify-start"
                data-testid={`button-add-${type.value}`}
              >
                <span className="mr-2">{type.icon}</span>
                {type.label}
              </Button>
            ))}
          </div>
        </div>

        {/* Section List */}
        <div className="space-y-2">
          <h4 className="font-medium">Current Sections ({sectionItems.length})</h4>
          
          {sectionItems.length === 0 ? (
            <div className="text-center py-8 text-muted-foreground border-2 border-dashed rounded-lg">
              <p>No sections added yet.</p>
              <p className="text-sm">Use the buttons above to add sections.</p>
            </div>
          ) : (
            <DragDropContext onDragEnd={handleDragEnd}>
              <Droppable droppableId="sections">
                {(provided) => (
                  <div
                    {...provided.droppableProps}
                    ref={provided.innerRef}
                    className="space-y-2"
                  >
                    {sectionItems.map((section, index) => (
                      <Draggable key={section.id} draggableId={section.id} index={index}>
                        {(provided, snapshot) => (
                          <div
                            ref={provided.innerRef}
                            {...provided.draggableProps}
                            className={`border rounded-lg p-3 bg-card ${
                              snapshot.isDragging ? 'shadow-lg' : ''
                            }`}
                          >
                            <div className="flex items-center justify-between">
                              <div className="flex items-center space-x-3">
                                <div
                                  {...provided.dragHandleProps}
                                  className="text-muted-foreground hover:text-foreground cursor-grab"
                                >
                                  <GripVertical className="w-4 h-4" />
                                </div>
                                
                                <Button
                                  variant="ghost"
                                  size="sm"
                                  onClick={() => toggleExpanded(section.id)}
                                  data-testid={`button-expand-${section.id}`}
                                >
                                  {section.isExpanded ? (
                                    <ChevronDown className="w-4 h-4" />
                                  ) : (
                                    <ChevronRight className="w-4 h-4" />
                                  )}
                                </Button>

                                <div className="flex items-center space-x-2">
                                  <span className="font-medium capitalize">
                                    {section.type}
                                  </span>
                                  <Badge variant="outline" className="text-xs">
                                    #{section.order + 1}
                                  </Badge>
                                </div>
                              </div>

                              <div className="flex items-center space-x-2">
                                <Button
                                  variant="ghost"
                                  size="sm"
                                  onClick={() => toggleVisibility(section.id)}
                                  data-testid={`button-toggle-visibility-${section.id}`}
                                >
                                  {section.isVisible ? (
                                    <Eye className="w-4 h-4" />
                                  ) : (
                                    <EyeOff className="w-4 h-4" />
                                  )}
                                </Button>
                                
                                <Button
                                  variant="ghost"
                                  size="sm"
                                  data-testid={`button-settings-${section.id}`}
                                >
                                  <Settings className="w-4 h-4" />
                                </Button>
                                
                                <Button
                                  variant="ghost"
                                  size="sm"
                                  onClick={() => removeSection(section.id)}
                                  data-testid={`button-remove-${section.id}`}
                                >
                                  <Trash2 className="w-4 h-4" />
                                </Button>
                              </div>
                            </div>

                            {section.isExpanded && (
                              <div className="mt-3 pt-3 border-t">
                                <div className="text-sm text-muted-foreground">
                                  <p><strong>Type:</strong> {section.type}</p>
                                  <p><strong>Order:</strong> {section.order + 1}</p>
                                  <p><strong>Visible:</strong> {section.isVisible ? 'Yes' : 'No'}</p>
                                </div>
                              </div>
                            )}
                          </div>
                        )}
                      </Draggable>
                    ))}
                    {provided.placeholder}
                  </div>
                )}
              </Droppable>
            </DragDropContext>
          )}
        </div>
      </CardContent>
    </Card>
  );
}