import { useState } from 'react';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { Button } from '@/components/ui/button';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Switch } from '@/components/ui/switch';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { Save, Loader2, Check } from 'lucide-react';
import { toast } from '@/hooks/use-toast';
import { apiRequest } from '@/lib/queryClient';
import type { WebsiteTemplate, UserTemplateCustomization, TemplateCustomization } from '@shared/schema';

interface SaveTemplateDialogProps {
  template: WebsiteTemplate;
  customization: TemplateCustomization;
  onSaveComplete?: (savedTemplate: UserTemplateCustomization) => void;
  children?: React.ReactNode;
  isOpen?: boolean;
  onOpenChange?: (open: boolean) => void;
}

const saveTemplateSchema = z.object({
  customizationName: z.string()
    .min(2, 'Template name must be at least 2 characters')
    .max(100, 'Template name must not exceed 100 characters'),
  description: z.string()
    .max(500, 'Description must not exceed 500 characters')
    .optional(),
  isFavorite: z.boolean().default(false),
  isTemplate: z.boolean().default(false),
});

type SaveTemplateFormData = z.infer<typeof saveTemplateSchema>;

export default function SaveTemplateDialog({
  template,
  customization,
  onSaveComplete,
  children,
  isOpen,
  onOpenChange
}: SaveTemplateDialogProps) {
  const [open, setOpen] = useState(false);
  const [saveMode, setSaveMode] = useState<'new' | 'overwrite'>('new');
  const [existingCustomizationId, setExistingCustomizationId] = useState<string | null>(null);
  const queryClient = useQueryClient();

  const isControlled = isOpen !== undefined && onOpenChange !== undefined;
  const dialogOpen = isControlled ? isOpen : open;
  const setDialogOpen = isControlled ? onOpenChange : setOpen;

  const form = useForm<SaveTemplateFormData>({
    resolver: zodResolver(saveTemplateSchema),
    defaultValues: {
      customizationName: `${template.name || 'Template'} - ${new Date().toLocaleDateString()}`,
      description: '',
      isFavorite: false,
      isTemplate: false,
    },
  });

  // Save template mutation
  const saveTemplateMutation = useMutation({
    mutationFn: async (data: SaveTemplateFormData) => {
      const payload = {
        templateId: template.id,
        customizationName: data.customizationName,
        configurations: customization,
        contentOverrides: customization.content,
        styleOverrides: {
          colors: customization.colors,
          fonts: customization.fonts,
          layout: customization.layout,
        },
        isTemplate: data.isTemplate,
        isFavorite: data.isFavorite,
      };

      if (data.description) {
        // Store description in configurations metadata
        payload.configurations = {
          ...payload.configurations,
          metadata: {
            ...(payload.configurations.metadata || {}),
            description: data.description,
          }
        };
      }

      // Use PUT for existing customizations, POST for new ones
      if (saveMode === 'overwrite' && existingCustomizationId) {
        return apiRequest('PUT', `/api/template-customizations/${existingCustomizationId}`, payload);
      } else {
        return apiRequest('POST', '/api/template-customizations', payload);
      }
    },
    onSuccess: (savedTemplate: UserTemplateCustomization) => {
      queryClient.invalidateQueries({ queryKey: ['/api/template-customizations'] });
      queryClient.invalidateQueries({ queryKey: ['/api/template-customizations', 'template', template.id] });
      
      toast({
        title: "Template Saved!",
        description: `"${form.getValues('customizationName')}" has been saved successfully.`,
        duration: 3000,
      });

      onSaveComplete?.(savedTemplate);
      setDialogOpen(false);
      form.reset();
    },
    onError: (error: any) => {
      console.error('Failed to save template:', error);
      toast({
        title: "Save Failed",
        description: error.message || "Failed to save template. Please try again.",
        variant: "destructive",
      });
    },
  });

  // Check for existing customization
  const checkExistingMutation = useMutation({
    mutationFn: async () => {
      try {
        return await apiRequest('GET', `/api/template-customizations/template/${template.id}`);
      } catch (error) {
        return null; // No existing customization
      }
    },
    onSuccess: (existingCustomization: UserTemplateCustomization | null) => {
      if (existingCustomization) {
        setSaveMode('overwrite');
        setExistingCustomizationId(existingCustomization.id);
        form.setValue('customizationName', existingCustomization.customizationName || '');
        
        // Extract description from metadata if it exists
        const configurations = existingCustomization.configurations as any || {};
        const description = configurations.metadata?.description;
        if (description) {
          form.setValue('description', description);
        }
        
        form.setValue('isFavorite', existingCustomization.isFavorite || false);
        form.setValue('isTemplate', existingCustomization.isTemplate || false);
      } else {
        setSaveMode('new');
        setExistingCustomizationId(null);
      }
    },
  });

  const onSubmit = (data: SaveTemplateFormData) => {
    saveTemplateMutation.mutate(data);
  };

  const handleOpenChange = (newOpen: boolean) => {
    setDialogOpen(newOpen);
    if (newOpen) {
      // Check for existing customization when opening
      checkExistingMutation.mutate();
    } else {
      form.reset();
      setSaveMode('new');
      setExistingCustomizationId(null);
    }
  };

  return (
    <Dialog open={dialogOpen} onOpenChange={handleOpenChange}>
      {children && (
        <DialogTrigger asChild>
          {children}
        </DialogTrigger>
      )}
      <DialogContent className="sm:max-w-[500px]" data-testid="dialog-save-template">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Save className="w-5 h-5" />
            {saveMode === 'overwrite' ? 'Update Template' : 'Save Template'}
          </DialogTitle>
          <DialogDescription>
            {saveMode === 'overwrite' 
              ? 'Update your existing template customization with the current settings.'
              : 'Save your current template customization for future use.'
            }
          </DialogDescription>
        </DialogHeader>

        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
            <FormField
              control={form.control}
              name="customizationName"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Template Name</FormLabel>
                  <FormControl>
                    <Input
                      placeholder="Enter template name..."
                      {...field}
                      data-testid="input-template-name"
                    />
                  </FormControl>
                  <FormDescription>
                    Choose a descriptive name for your template.
                  </FormDescription>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="description"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Description (Optional)</FormLabel>
                  <FormControl>
                    <Textarea
                      placeholder="Describe what makes this template special..."
                      className="min-h-[80px]"
                      {...field}
                      data-testid="textarea-template-description"
                    />
                  </FormControl>
                  <FormDescription>
                    Add details about the template's purpose or design.
                  </FormDescription>
                  <FormMessage />
                </FormItem>
              )}
            />

            <div className="flex flex-col space-y-4">
              <FormField
                control={form.control}
                name="isFavorite"
                render={({ field }) => (
                  <FormItem className="flex flex-row items-center justify-between rounded-lg border p-3">
                    <div className="space-y-0.5">
                      <FormLabel className="text-base">
                        Mark as Favorite
                      </FormLabel>
                      <FormDescription>
                        Add to your favorites for quick access.
                      </FormDescription>
                    </div>
                    <FormControl>
                      <Switch
                        checked={field.value}
                        onCheckedChange={field.onChange}
                        data-testid="switch-template-favorite"
                      />
                    </FormControl>
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="isTemplate"
                render={({ field }) => (
                  <FormItem className="flex flex-row items-center justify-between rounded-lg border p-3">
                    <div className="space-y-0.5">
                      <FormLabel className="text-base">
                        Save as Template
                      </FormLabel>
                      <FormDescription>
                        Make this available for others to use.
                      </FormDescription>
                    </div>
                    <FormControl>
                      <Switch
                        checked={field.value}
                        onCheckedChange={field.onChange}
                        data-testid="switch-save-as-template"
                      />
                    </FormControl>
                  </FormItem>
                )}
              />
            </div>

            <div className="flex justify-end space-x-3 pt-4">
              <Button
                type="button"
                variant="outline"
                onClick={() => handleOpenChange(false)}
                disabled={saveTemplateMutation.isPending}
                data-testid="button-cancel-save"
              >
                Cancel
              </Button>
              <Button
                type="submit"
                disabled={saveTemplateMutation.isPending}
                className="min-w-[120px]"
                data-testid="button-confirm-save"
              >
                {saveTemplateMutation.isPending ? (
                  <>
                    <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                    Saving...
                  </>
                ) : saveTemplateMutation.isSuccess ? (
                  <>
                    <Check className="w-4 h-4 mr-2" />
                    Saved
                  </>
                ) : (
                  <>
                    <Save className="w-4 h-4 mr-2" />
                    {saveMode === 'overwrite' ? 'Update' : 'Save'}
                  </>
                )}
              </Button>
            </div>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  );
}