import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Slider } from '@/components/ui/slider';
import { Button } from '@/components/ui/button';
import { RefreshCw } from 'lucide-react';

interface Layout {
  spacing: string;
  maxWidth: string;
  gridColumns: number;
}

interface LayoutCustomizerProps {
  layout: Layout;
  onChange: (layout: Layout) => void;
}

const spacingOptions = [
  { value: 'compact', label: 'Compact' },
  { value: 'normal', label: 'Normal' },
  { value: 'relaxed', label: 'Relaxed' },
  { value: 'loose', label: 'Loose' }
];

const maxWidthOptions = [
  { value: '960px', label: 'Small (960px)' },
  { value: '1200px', label: 'Medium (1200px)' },
  { value: '1440px', label: 'Large (1440px)' },
  { value: '100%', label: 'Full Width' }
];

export default function LayoutCustomizer({ layout, onChange }: LayoutCustomizerProps) {
  const handleLayoutChange = (key: keyof Layout, value: string | number) => {
    onChange({
      ...layout,
      [key]: value,
    });
  };

  const resetLayout = () => {
    onChange({
      spacing: 'normal',
      maxWidth: '1200px',
      gridColumns: 12
    });
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center justify-between">
          Layout Settings
          <Button
            variant="ghost"
            size="sm"
            onClick={resetLayout}
            data-testid="button-reset-layout"
          >
            <RefreshCw className="w-4 h-4" />
          </Button>
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-6">
        <div className="space-y-2">
          <Label>Spacing</Label>
          <Select
            value={layout.spacing}
            onValueChange={(value) => handleLayoutChange('spacing', value)}
          >
            <SelectTrigger data-testid="select-spacing">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              {spacingOptions.map((option) => (
                <SelectItem key={option.value} value={option.value}>
                  {option.label}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        <div className="space-y-2">
          <Label>Maximum Width</Label>
          <Select
            value={layout.maxWidth}
            onValueChange={(value) => handleLayoutChange('maxWidth', value)}
          >
            <SelectTrigger data-testid="select-max-width">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              {maxWidthOptions.map((option) => (
                <SelectItem key={option.value} value={option.value}>
                  {option.label}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        <div className="space-y-2">
          <div className="flex items-center justify-between">
            <Label>Grid Columns</Label>
            <span className="text-sm text-muted-foreground">{layout.gridColumns}</span>
          </div>
          <Slider
            value={[layout.gridColumns]}
            onValueChange={([value]) => handleLayoutChange('gridColumns', value)}
            min={6}
            max={24}
            step={2}
            className="w-full"
            data-testid="slider-grid-columns"
          />
        </div>

        <div className="pt-4 border-t">
          <h4 className="font-medium mb-3">Preview</h4>
          <div className="bg-muted/30 rounded p-4">
            <div 
              className="grid gap-2 mx-auto"
              style={{ 
                maxWidth: layout.maxWidth === '100%' ? '100%' : layout.maxWidth,
                gridTemplateColumns: `repeat(${Math.min(layout.gridColumns, 4)}, 1fr)`
              }}
            >
              {Array.from({ length: Math.min(layout.gridColumns, 4) }).map((_, i) => (
                <div key={i} className="bg-primary/20 h-8 rounded"></div>
              ))}
            </div>
          </div>
        </div>
      </CardContent>
    </Card>
  );
}