import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Slider } from '@/components/ui/slider';
import { Button } from '@/components/ui/button';
import { RefreshCw } from 'lucide-react';

interface FontSizes {
  h1: string;
  h2: string;
  h3: string;
  body: string;
}

interface Fonts {
  heading: string;
  body: string;
  sizes: FontSizes;
}

interface FontCustomizerProps {
  fonts: Fonts;
  onChange: (fonts: Fonts) => void;
}

const fontOptions = [
  'Inter', 'Roboto', 'Open Sans', 'Lato', 'Poppins', 'Montserrat',
  'Source Sans Pro', 'Nunito Sans', 'Playfair Display', 'Merriweather'
];

export default function FontCustomizer({ fonts, onChange }: FontCustomizerProps) {
  const handleFontChange = (key: keyof Omit<Fonts, 'sizes'>, value: string) => {
    onChange({
      ...fonts,
      [key]: value,
    });
  };

  const handleSizeChange = (key: keyof FontSizes, value: string) => {
    onChange({
      ...fonts,
      sizes: {
        ...fonts.sizes,
        [key]: value,
      },
    });
  };

  const resetFonts = () => {
    onChange({
      heading: 'Inter',
      body: 'Inter',
      sizes: {
        h1: '2.5rem',
        h2: '2rem',
        h3: '1.5rem',
        body: '1rem'
      }
    });
  };

  const pxToRem = (px: number) => `${px / 16}rem`;
  const remToPx = (rem: string) => parseFloat(rem.replace('rem', '')) * 16;

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center justify-between">
          Typography
          <Button
            variant="ghost"
            size="sm"
            onClick={resetFonts}
            data-testid="button-reset-fonts"
          >
            <RefreshCw className="w-4 h-4" />
          </Button>
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-6">
        <div className="space-y-4">
          <div className="space-y-2">
            <Label>Heading Font</Label>
            <Select
              value={fonts.heading}
              onValueChange={(value) => handleFontChange('heading', value)}
            >
              <SelectTrigger data-testid="select-heading-font">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                {fontOptions.map((font) => (
                  <SelectItem key={font} value={font}>
                    {font}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div className="space-y-2">
            <Label>Body Font</Label>
            <Select
              value={fonts.body}
              onValueChange={(value) => handleFontChange('body', value)}
            >
              <SelectTrigger data-testid="select-body-font">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                {fontOptions.map((font) => (
                  <SelectItem key={font} value={font}>
                    {font}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>

        <div className="space-y-4">
          <h4 className="font-medium">Font Sizes</h4>
          
          {Object.entries(fonts.sizes).map(([key, value]) => (
            <div key={key} className="space-y-2">
              <div className="flex items-center justify-between">
                <Label className="capitalize">{key === 'body' ? 'Body Text' : key}</Label>
                <span className="text-sm text-muted-foreground">{value}</span>
              </div>
              <Slider
                value={[remToPx(value)]}
                onValueChange={([px]) => handleSizeChange(key as keyof FontSizes, pxToRem(px))}
                min={12}
                max={72}
                step={2}
                className="w-full"
                data-testid={`slider-font-size-${key}`}
              />
            </div>
          ))}
        </div>
      </CardContent>
    </Card>
  );
}